/*
 * Copyright 2001    by TriplePoint, Inc. (timg@tpi.com)
 * Copyright 2001    by Symbol Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * This file implements the /proc filesystem for the Spectrum24 802.11 driver. Each adapter
 * gets an entry whose name is encoded with its ethernet interface name, e.g.,
 * /proc/spectrum24t_eth0. A super-user application can read/write this interface with
 * SCORCH_RW structures.
 * The IOCTL interface is also implemented
 * here. It is used primarily for AP mode configuration.
 *
 * The IOCTL commands are as follows:
 *
 *  _SCORCH_IOCTL_CONNECT
 *  _SCORCH_IOCTL_DISCONNECT
 *  _SCORCH_IOCTL_WRITE_RID
 *  _SCORCH_IOCTL_READ_RID
 *
 */
#include "Spectrum24tApi.h"
#include "Spectrum24t.h"
#include <linux/poll.h>

#if S24_PROC_SYS

#if DBG
static drv_info_t *DbgInfo = &S24tInfo;
#endif

DECLARE_WAIT_QUEUE_HEAD(ApRecvWaitQueue);

/* The module's file functions ********************** */

static dev_link_t **dev_list=NULL;

PS24tPrivate
S24tFindContextFromInode(
    struct inode *inode,
    dev_link_t    **link
    )
{
    DBG_FUNC("S24tFindContextFromInode")

    PS24tPrivate S24t=NULL;

    DEVICE  *dev;
    dev_link_t    **linkp;

    /* Locate device structure */
    for (linkp = dev_list; *linkp; linkp = &(*linkp)->next)
    {
        if ((dev = (*linkp)->priv))
        {
            if ((S24t = dev->priv))
            {
                if (S24t->ProcFile->low_ino == (unsigned short)inode->i_ino)
                {
                    if (link)
                    {
                        *link = *linkp;
                    }
                    return(S24t);
                }
            }
        }
    }

    DBG_ASSERT(0);
    return(NULL);
}

PS24tPrivate
S24tFindContextFromFile(
    struct file *file
    )
{
    return( S24tFindContextFromInode(file->f_dentry->d_inode,NULL) );
}

void S24tProcQueueRx(PS24tPrivate S24t, struct sk_buff *skb)
{
    int flags;

    local_irq_save(flags);
    __skb_queue_tail(&S24t->ApRxQueue,skb);
    local_irq_restore(flags);

    wake_up_interruptible(&ApRecvWaitQueue);
}

/* Since we use the file operations struct, we can't
 * use the special proc output provisions - we have to
 * use a standard read function, which is this function */
static ssize_t S24tProcFileSysOutputToUser(
    struct file *file,   /* The file read */
    char *buf, /* The buffer to put data to (in the
                * user segment) */
    size_t len,  /* The length of the buffer */
    loff_t *offset) /* Offset in the file - ignore */
{
    DBG_FUNC("S24tProcFileSysOutputToUser")
    PS24tPrivate S24t;
    struct sk_buff *skb;
    int flags;
    int attempts=1;

    DBG_ENTER(DbgInfo);

    if (!(S24t = S24tFindContextFromFile(file)))
    {
        DBG_PRINT("S24tProcFileSysOutputToUser could not S24tFindContextFromFile()\n");
        DBG_LEAVE(DbgInfo);
        return(-EIO);
    }

    local_irq_save(flags);

    /*
     * Wait for 3 seconds once if the initial attempt comes up empty.
     */
    while (
           (!(skb = __skb_dequeue(&S24t->ApRxQueue)))
           &&
           attempts
          )
    {
        /*
         * Sleep for a time.
         */
        interruptible_sleep_on_timeout(&ApRecvWaitQueue,3*HZ);
        cli();
        attempts--;
    }

    local_irq_restore(flags);

    if (!skb)
    {
        DBG_LEAVE(DbgInfo);
        return(0);
    }

    if (
        verify_area(VERIFY_WRITE,buf,skb->len)
        ||
        (skb->len > len)
       )
    {
        printk(KERN_ERR "spectrum24t could not VERIFY_WRITE len %d\n",skb->len);
        DEV_KFREE_SKB(skb);
        DBG_LEAVE(DbgInfo);
        return(-EIO);
    }

    copy_to_user(buf,skb->data,skb->len);
    len = skb->len;
    DEV_KFREE_SKB(skb);

    DBG_PROC(DbgInfo,"S24tProcFileSysOutputToUser inode %u len %d\n",(uint)file->f_dentry->d_inode->i_ino,len);
    DBG_LEAVE(DbgInfo);
    return(len);
}

void S24tProcStartTx(PS24tPrivate S24t)
{
    DBG_FUNC("S24tProcStartTx")
    int             flags;
    struct sk_buff  *skb;

    DBG_ASSERT(S24t->ApMode);

    local_irq_save(flags);
    if (S24tTxReady(S24t))
    {
        if ((skb = __skb_dequeue(&S24t->ApTxQueue)))
        {
#if DBG
            DBG_ASSERT(S24tTx(S24t,skb->data,skb->len) >= 0);
#else
            S24tTx(S24t,skb->data,skb->len);
#endif
            DEV_KFREE_SKB(skb);
        }
    }
    local_irq_restore(flags);
}

/* This function receives input from the user when the
 * user writes to the /proc file. */
static ssize_t S24tProcFileSysInputFromUser(
    struct file *file,   /* The file itself */
    const char *buf,     /* The buffer with input */
    size_t length,       /* The buffer's length */
    loff_t *offset)      /* offset to file - ignore */
{
    DBG_FUNC("S24tProcFileSysInputFromUser")
    PS24tPrivate    S24t;
    int             flags;
    struct sk_buff *skb;
    unsigned long   len = (unsigned long)length;

    DBG_ENTER(DbgInfo);
    DBG_PROC(DbgInfo,"S24tProcFileSysInputFromUser inode %u len %d\n",(uint)file->f_dentry->d_inode->i_ino,length);

    if (!(S24t = S24tFindContextFromFile(file)))
    {
        DBG_PRINT("S24tProcFileSysInputFromUser could not S24tFindContextFromFile()\n");
        DBG_LEAVE(DbgInfo);
        return(0);
    }

    if (
        (length <= 0)
        ||
        verify_area(VERIFY_READ,buf,len)
       )
    {
        printk(KERN_ERR "spectrum24t could not VERIFY_READ len %lu\n",len);
        DBG_LEAVE(DbgInfo);
        return(-EIO);
    }

    skb = ALLOC_SKB(len);
    if (!skb)
    {
        printk(KERN_ERR "spectrum24t could not allocate SKB len %lu\n",len);
        DBG_LEAVE(DbgInfo);
        return(-ENOMEM);
    }

    copy_from_user(skb->data,buf,len);
    skb_put(skb,len);

    local_irq_save(flags);
    __skb_queue_tail(&S24t->ApTxQueue,skb);
    local_irq_restore(flags);

    S24tProcStartTx(S24t);

    DBG_LEAVE(DbgInfo);
    return(length);
}


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)

/* This function decides whether to allow an operation
 * (return zero) or not allow it (return a non-zero
 * which indicates why it is not allowed).
 *
 * The operation can be one of the following values:
 * 0 - Execute (run the "file" - meaningless in our case)
 * 2 - Write (input to the kernel module)
 * 4 - Read (output from the kernel module)
 *
 * This is the real function that checks file
 * permissions. The permissions returned by ls -l are
 * for referece only, and can be overridden here.
 */
static int S24tPermission(struct inode *inode, int op)
{
    DBG_PROC(DbgInfo,"S24tPermission inode %u\n", (uint)inode->i_ino);

  /* We allow everybody to read from our module, but
   * only root (uid 0) may write to it */
  if (op == 4 || (op == 2 && current->euid == 0))
    return 0;

  /* If it's anything else, access is denied */
  return -EACCES;
}

#endif

static int
S24tProcFileSysIoctl(
    struct inode *inode,
    struct file *file,
    unsigned int cmd,
    unsigned long arg
    )
{
    DBG_FUNC("S24tProcFileSysIoctl")
    PS24tPrivate    S24t;
    PSCORCH_RID     rid = (PSCORCH_RID)arg;
    int             flags;
    int             Status = 0;

    DBG_ENTER(DbgInfo);

    if (!(S24t = S24tFindContextFromInode(inode,NULL)))
    {
        DBG_PRINT("Bogus inode %d\n",(uint)inode->i_ino);
        DBG_LEAVE(DbgInfo);
        return(-EACCES);
    }
    if (current->euid)
    {
        printk(KERN_ERR "spectrum24t: Only root access on %s ioctl\n",S24t->ProcName);
        DBG_LEAVE(DbgInfo);
        return(-EACCES);
    }

    DBG_ASSERT(sizeof(SCORCH_RID) == (SCORCH_RID_DATA_LEN+4));
    DBG_ASSERT(sizeof(SCORCH_DEVICE_HEADER) == 22);
    DBG_ASSERT(sizeof(SCORCH_RW) >= (sizeof(SCORCH_DEVICE_HEADER)+1514));

    switch (cmd)
    {
    case _SCORCH_IOCTL_WRITE_RID:
    case _SCORCH_IOCTL_READ_RID:
        if (
            verify_area(VERIFY_READ,rid,2*sizeof(__u16))
           )
        {
            DBG_PRINT("S24tProcFileSysIoctl: Cannot verify area for read/write %d bytes.\n",2*sizeof(__u16));
            return(-EIO);
        }
        if (
            verify_area(VERIFY_READ,rid,RID_DATA_LEN(rid))
            ||
            verify_area(VERIFY_WRITE,rid,RID_DATA_LEN(rid))
           )
        {
            DBG_PRINT("S24tProcFileSysIoctl: Cannot verify area for read/write %d bytes.\n",RID_DATA_LEN(rid));
            return(-EIO);
        }
        break;
    default:
        break;
    }

    local_irq_save(flags);

    switch (cmd)
    {
    case _SCORCH_IOCTL_CONNECT:
        DBG_PROC(DbgInfo,"_SCORCH_IOCTL_CONNECT ->\n");

        /*
         * Always enables AP mode.
         */
        Status = S24tEnable(S24t,1);
        DBG_PROC(DbgInfo,"_SCORCH_IOCTL_CONNECT <- Status %d\n",Status);
        break;
    case _SCORCH_IOCTL_DISCONNECT:
        DBG_PROC(DbgInfo,"_SCORCH_IOCTL_DISCONNECT ->\n");

        Status = S24tDisable(S24t,1);
        DBG_PROC(DbgInfo,"_SCORCH_IOCTL_DISCONNECT <- Status %d\n",Status);
        break;
    case _SCORCH_IOCTL_WRITE_RID:
        DBG_PROC(DbgInfo,"_SCORCH_IOCTL_WRITE_RID %04x data len %04x\n",
            rid->rid,
            RID_DATA_LEN(rid)
        );
        Status =
        S24tWriteRid(
            S24t,
            rid->rid,
            rid->data,
            RID_DATA_LEN(rid)
        );
        DBG_PROC(DbgInfo,"_SCORCH_IOCTL_WRITE_RID Status %d\n",Status);
        break;

    case _SCORCH_IOCTL_READ_RID:
        DBG_PROC(DbgInfo,"_SCORCH_IOCTL_READ_RID %04x data len %04x\n",
            rid->rid,
            RID_DATA_LEN(rid)
        );
        Status =
        S24tReadRid(
            S24t,
            rid->rid,
            rid->data,
            RID_DATA_LEN(rid)
        );
        DBG_PROC(DbgInfo,"_SCORCH_IOCTL_READ_RID Status %d\n",Status);
        break;

    default:
        Status = (-EIO);
        break;
    }

    local_irq_restore(flags);

    DBG_LEAVE(DbgInfo);
    return(Status);
}



/* The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count. */
static int S24tProcFileSysOpen(struct inode *inode, struct file *file)
{
    DBG_FUNC("S24tProcFileSysOpen")
    PS24tPrivate S24t;
    dev_link_t  *link;

    DBG_ENTER(DbgInfo);
    DBG_PROC(DbgInfo,"S24tProcFileSysOpen inode %u\n", (uint)inode->i_ino);

    if (!(S24t = S24tFindContextFromInode(inode,&link)))
    {
        DBG_LEAVE(DbgInfo);
        return -EACCES;
    }

    /*
     * Do not allow side doors while the main interface is functioning.
     */
    if (link->open)
    {
        printk(KERN_ERR "spectrum24t: proc file system is unavailable while network interfaces are open.\n");
        DBG_LEAVE(DbgInfo);
        return -EACCES;
    }

    if (current->euid)
    {
        printk(KERN_ERR "spectrum24t: Only root access allowed on %s\n",S24t->ProcName);
        DBG_LEAVE(DbgInfo);
        return(-EACCES);
    }

    /*
     * Only one owner at a time.
     */
    if (S24t->ProcSysOpen)
    {
        printk(KERN_ERR "spectrum24t: Only one open at a time allowed on %s\n",S24t->ProcName);
        DBG_LEAVE(DbgInfo);
        return -EACCES;
    }

    S24t->ProcSysOpen = 1;

    MOD_INC_USE_COUNT;

    DBG_LEAVE(DbgInfo);
    return 0;
}


/* The file is closed - again, interesting only because
 * of the reference count. */
static int S24tProcFileSysClose(struct inode *inode, struct file *file)
{
    DBG_FUNC("S24tProcFileSysClose")
    PS24tPrivate    S24t;
    int             flags;

    DBG_ENTER(DbgInfo);
    DBG_PROC(DbgInfo,"S24tProcFileSysClose inode %u\n", (uint)inode->i_ino);

    if (!(S24t = S24tFindContextFromFile(file)))
    {
        DBG_LEAVE(DbgInfo);
        return -EACCES;
    }

    local_irq_save(flags);

    __skb_queue_purge(&S24t->ApTxQueue);
    __skb_queue_purge(&S24t->ApRxQueue);

    S24tDisable(S24t,1);

    local_irq_restore(flags);

    S24t->ProcSysOpen = 0;

    MOD_DEC_USE_COUNT;

    DBG_LEAVE(DbgInfo);
    return 0;  /* success */
}

static unsigned int S24tProcFileSysPoll(struct file *file, poll_table *wait)
{
    DBG_FUNC("S24tProcFileSysPoll")

    PS24tPrivate    S24t;
    unsigned int mask = POLLOUT | POLLWRNORM;

    DBG_ENTER(DbgInfo);

    if (!(S24t = S24tFindContextFromFile(file)))
    {
        DBG_LEAVE(DbgInfo);
        return -EACCES;
    }

    poll_wait(file, &ApRecvWaitQueue, wait);

    if (skb_queue_len(&S24t->ApRxQueue))
    {
        mask |= (POLLIN | POLLRDNORM);
    }

    DBG_LEAVE(DbgInfo);
    return(mask);
}

/* Structures to register as the /proc file, with
 * pointers to all the relevant functions. ********** */



/* File operations for our proc file. This is where we
 * place pointers to all the functions called when
 * somebody tries to do something to our file. NULL
 * means we don't want to deal with something. */
static struct file_operations S24tFileOps =
  {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
    owner:  THIS_MODULE,
    read:   S24tProcFileSysOutputToUser,
    write:  S24tProcFileSysInputFromUser,
    ioctl:  S24tProcFileSysIoctl,
    open:   S24tProcFileSysOpen,
    release:S24tProcFileSysClose,
    poll:   S24tProcFileSysPoll
#else
    NULL,  /* lseek */
    S24tProcFileSysOutputToUser,  /* "read" from the file */
    S24tProcFileSysInputFromUser,   /* "write" to the file */
    NULL,  /* readdir */
    NULL,  /* select */
    S24tProcFileSysIoctl,  /* ioctl */
    NULL,  /* mmap */
    S24tProcFileSysOpen,    /* Somebody opened the file */
    NULL,   /* flush, added here in version 2.2 */
    S24tProcFileSysClose,    /* Somebody closed the file */
    /* etc. etc. etc. (they are all given in
     * /usr/include/linux/fs.h). Since we don't put
     * anything here, the system will keep the default
     * data, which in Unix is zeros (NULLs when taken as
     * pointers). */
#endif
  };


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)

/* Inode operations for our proc file. We need it so
 * we'll have some place to specify the file operations
 * structure we want to use, and the function we use for
 * permissions. It's also possible to specify functions
 * to be called for anything else which could be done to
 * an inode (although we don't bother, we just put
 * NULL). */
static struct inode_operations S24tInodeOps =
  {
    &S24tFileOps,
    NULL, /* create */
    NULL, /* lookup */
    NULL, /* link */
    NULL, /* unlink */
    NULL, /* symlink */
    NULL, /* mkdir */
    NULL, /* rmdir */
    NULL, /* mknod */
    NULL, /* rename */
    NULL, /* readlink */
    NULL, /* follow_link */
    NULL, /* readpage */
    NULL, /* writepage */
    NULL, /* bmap */
    NULL, /* truncate */
    S24tPermission /* check for permissions */
  };

/* Directory entry */
static struct proc_dir_entry S24tProcFile =
  {
    0, /* Inode number - ignore, it will be filled by
        * proc_register[_dynamic] */
    0, /* Length of the file name */
    NULL, /* The file name */
    S_IFREG | S_IRUGO | S_IWUSR,
    /* File mode - this is a regular file which
     * can be read by its owner, its group, and everybody
     * else. Also, its owner can write to it.
     *
     * Actually, this field is just for reference, it's
     * S24tPermission that does the actual check. It
     * could use this field, but in our implementation it
     * doesn't, for simplicity. */
    1,  /* Number of links (directories where the
         * file is referenced) */
    0, 0,  /* The uid and gid for the file -
            * we give it to root */
    0, /* The size of the file reported by ls. */
    &S24tInodeOps,
  };
#endif

/*
 * Register /proc/spectrum24t_ethX. Returns 0 on success.
 */
int
S24tRegisterProcSys(
    PS24tPrivate    S24t,
    dev_link_t      **t_dev_list
    )
{
    DBG_FUNC("S24tRegisterProcSys")

    int Status=0;

    DBG_ENTER(DbgInfo);
    sprintf(S24t->ProcName,S24T_PROCSYS_NAME_FORMAT,S24t->dev->name);

    DBG_PROC(DbgInfo,"S24tRegisterProcSys registering %s\n",S24t->ProcName);

    skb_queue_head_init(&S24t->ApTxQueue);
    skb_queue_head_init(&S24t->ApRxQueue);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
    if (!(S24t->ProcFile=create_proc_entry(S24t->ProcName,S_IRUGO | S_IWUSR,&proc_root)))
    {
        Status = -EINVAL;
    }
    else
    {
        S24t->ProcFile->owner = THIS_MODULE;
        S24t->ProcFile->proc_fops = &S24tFileOps;
    }
#else
    S24t->ProcFile = &S24t->ProcFileVal;
    *S24t->ProcFile = S24tProcFile;
    S24t->ProcFile->name = S24t->ProcName;
    S24t->ProcFile->namelen = strlen(S24t->ProcName);

    Status = proc_register(&proc_root, S24t->ProcFile);
#endif

    if (Status)
    {
        DBG_PRINT("proc_register failed on %s\n",S24t->ProcName);
    }
    else
    {
        /*
         * This is how we find our context when opened.
         */
        dev_list = t_dev_list;
        DBG_PROC(DbgInfo,"proc_register returned inode %u\n",(uint)S24t->ProcFile->low_ino);
    }

    DBG_LEAVE(DbgInfo);
    return(Status);
}

void
S24tUnRegisterProcSys(
    PS24tPrivate   S24t
    )
{
    DBG_FUNC("S24tUnRegisterProcSys")

    DBG_ASSERT(!S24t->ProcSysOpen);

    DBG_ENTER(DbgInfo);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0)
    remove_proc_entry(S24t->ProcName,&proc_root);
#else
    if (S24t->ProcFile->low_ino)
    {
        DBG_PROC(DbgInfo,"S24tUnRegisterProcSys unregistering %s inode %u\n",S24t->ProcName,(uint)S24t->ProcFile->low_ino);
        proc_unregister(&proc_root, S24t->ProcFile->low_ino);
        S24t->ProcFile->low_ino = 0;
    }
#endif

    S24t->ProcFile = NULL;

    DBG_PROC(DbgInfo,"S24tUnRegisterProcSys %s\n",S24t->ProcName);
    DBG_LEAVE(DbgInfo);
}

#endif

