#!/usr/bin/ksh
#
# NAME:  do-what.ksh levelListFile includeFileBaseName
#
# SAMPLE INVOCATION:
#
#   do-what.ksh master.list what-info
#
# PURPOSE:
#  To prepare the following include files with the information 
#  of the serviceability strings specified in the levelListFile argument
#  (such as master.list).
#    * include header file, such as what-info.h 
#    * include source file, such as what-info.c 
#    * include source file for standalone level application,
#      such as what-info-level.c 
#
# CUSTOMIZATION NOTES:
#  * See the sections labeled "CUSTOMIZATION REQUIRED". 
#  * You may want to customize this script to suit your needs. For example:
#    * Replace the value for the "keyword" variable to identify 
#      your application.
#
#*********************************************************************/

# CUSTOMIZATION NEEDED.
# Set the keyword to be used during what and grep

keyword="myApp"

# Verify the input parameters

if [ "$#" -eq 2 ]
then
  levelListFile=$1
  includeFileBaseName=$2
  includeHeaderFile=`echo $includeFileBaseName`".h"
  includeSourceFile=`echo $includeFileBaseName`".c"
  includeSourceLevelFile=`echo $includeFileBaseName`"-level.c"
  print "do-what.ksh begin"
  print " -> file with level information: $levelListFile "
  print " -> include header file:         $includeHeaderFile"
  print " -> include source file:         $includeSourceFile" 
  print " -> include source level file:   $includeSourceLevelFile" 
else
  print "*** Error. Invalid number of arguments"
  print "Usage: do-what.ksh levelListFile includeFileBaseName"
  print ""
  exit 1
fi

#
# Generating the include header file
#

print ""
print "do-what.ksh: generating include header file: $includeHeaderFile"

rm $includeHeaderFile 2>/dev/null
print "/* Name:	 $includeHeaderFile */"                             >> $includeHeaderFile
print "/* Purpose: This file is generated during build time.    */" >> $includeHeaderFile
print "/*  It has the 'build-time' serviceability information.  */" >> $includeHeaderFile
print "/*  to be displayed by using the what utility.           */" >> $includeHeaderFile
print ""                                                            >> $includeHeaderFile
print "#ifndef _WHAT_INFO_H_"                                       >> $includeHeaderFile
print "#define _WHAT_INFO_H_"                                       >> $includeHeaderFile

while read line
do
   name=`echo $line | cut -d' ' -f1`    # Get first field
   value=`echo $line | cut -d' ' -f2-`  # Get everything after the first field
   
   stringValue=`echo "char $name[] = \"@(#) $keyword $value\";"`     
   print "$stringValue"                                             >> $includeHeaderFile
done < $levelListFile

print ""                                                            >> $includeHeaderFile
print "#endif"                                                      >> $includeHeaderFile
print ""                                                            >> $includeHeaderFile
print "/* end of file */"                                           >> $includeHeaderFile

#
# Generating the include generic source file
#

print ""
print "do-what.ksh: generating include generic source file: $includeSourceFile"

rm $includeSourceFile 2>/dev/null

print "/* Name:	 $includeSourceFile */"                             >> $includeSourceFile
print "/* Purpose: This file is generated during build time.     */" >> $includeSourceFile
print "/*   It has the 'build-time' serviceability information.  */" >> $includeSourceFile
print "/*   to be displayed by using the what utility.           */" >> $includeSourceFile
print "/* Note: It is necessary to declare and use the variables */" >> $includeSourceFile
print "/*       in order to fool the compiler; otherwise, the    */" >> $includeSourceFile
print "/*       compiler will remove them from the executable.   */" >> $includeSourceFile


print ""                                                             >> $includeSourceFile
print "int dummy=3;      /* Set to a number different than -7077 */" >> $includeSourceFile
print "char dummyString[2000]; "                                     >> $includeSourceFile
print ""                                                             >> $includeSourceFile

while read line
do
   name=`echo $line | cut -d' ' -f1`    # Get first field
   
   stringValue="strcpy(dummyString,$name);"     
   print "$stringValue"                                              >> $includeSourceFile
done < $levelListFile

print ""                                                             >> $includeSourceFile
print "if (dummy == -7077) {"                                        >> $includeSourceFile
print "   printf(\"%s\",dummyString);"                               >> $includeSourceFile
print "}"                                                            >> $includeSourceFile

print ""                                                             >> $includeSourceFile
print "/* end of file */"                                           >> $includeSourceFile


#
# Generating the include level source file
#

print ""
print "do-what.ksh: generating include level source file: $includeSourceLevelFile"

rm $includeSourceLevelFile 2>/dev/null

print "/* Name:	 $includeSourceLevelFile */"                             >> $includeSourceLevelFile
print "/* Purpose: This file is generated during build time.     */" >> $includeSourceLevelFile
print "/*   It has the 'build-time' serviceability information   */" >> $includeSourceLevelFile
print "/*   to be displayed by the standalone level application. */" >> $includeSourceLevelFile
print ""                                                             >> $includeSourceLevelFile

while read line
do
   name=`echo $line | cut -d' ' -f1`    # Get first field
   
   stringValue="printf(\"%s\\\n\",$name);"     
   print "$stringValue"                                              >> $includeSourceLevelFile
done < $levelListFile

print ""                                                             >> $includeSourceLevelFile
print "/* end of file */"                                           >> $includeSourceLevelFile

print "do-what.ksh: done!"
print ""

exit 0

# end of file
