/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*    Kurt Maerker       Kurt_Maerker@vnet.ibm.com                          */
/*                                                              03/03/97    */
/*  Yodel_s.cmd  -  A TCP/IP Server with sessions                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/

aYodelServer = .YodelServer~new(1923)  /* Create a yodel server at port 1923*/
aYodelServer~startAccepting            /* Listen at socket for clients      */    

/****************************************************************************/

::REQUIRES "servers.frm"               /* Load the servers framework        */   

/*--------------------------------------------------------------------------*/
/* Repeater Class definition                                                */
/*--------------------------------------------------------------------------*/
::CLASS YodelServer SUBCLASS tcpServer PUBLIC

/*--------------------------------------------------------------------------*/
::METHOD NewClient                     /* Over-write superclass method      */    
  use arg cSocket

  aYodelClient = .Yodeller~new(cSocket)/* Create a yodeller instance        */
  aYodelClient~Listen                  /* then listen to any words          */
  aYodelClient~Yodel                   /* and then yodel them!              */


/*--------------------------------------------------------------------------*/
/* Yodeller Class definition                                                */
/*--------------------------------------------------------------------------*/
::CLASS Yodeller                       /* One for each client               */    

/*--------------------------------------------------------------------------*/
::METHOD init                          /* Initialize a yodeller instance    */    
  expose mySocket myName theWords 
  use arg mySocket

  myName = mySocket~HostName           /* This is my name                   */    
  theWords = "Yoda-LADY-oh! "          /* The initial words                 */

/*--------------------------------------------------------------------------*/
::METHOD Listen UNGUARDED 
  expose mySocket myName theWords 
  reply                                /* Frees up for other concurrent work*/ 
                                       /* Process the rest in another thread*/
  say "  * Yodeller started listening (client:" myName ||")"

                                       /* Repeat until the words are 'quit' */    
  do until theWords~word(1)~translate = "QUIT" 
    theWords = mySocket~ReceiveData
    if \mySocket~stillOpen then leave  /* if problems, end client session   */ 
  end
  say "  * Yodeller ended listening (client:" myName ||")"
  
/*--------------------------------------------------------------------------*/
::METHOD Yodel UNGUARDED
  expose mySocket myName theWords 
  reply                                /* Frees up for other concurrent work*/ 

  say "  * Yodeller started yodelling (client:" myName ||")"
  myWaiter = .Waiter~new

  do forever                           /* Yodel loop                        */   
    myWaiter~wait(2)                   /* Wait for 2 seconds, if not 'quit' */
    if theWords~word(1)~translate = 'QUIT' then leave 
    mySocket~SendData(theWords)        /* then send the words               */
    if \mySocket~stillOpen then leave  /* if problems, end client session   */ 
  end

  say "  * Yodeller ended yodelling (client:" myName ||")"
  self~shutdown                        /* Now shutdown the client           */

/*--------------------------------------------------------------------------*/
::METHOD shutdown                      /* Shutdown the client               */
  expose mySocket myName theWords    

  mySocket~SendData(">>>End_of_session<<<")
  say " ** Shutdown host:" myName 
  mySocket~Close                       /* Close this client socket          */ 


/*--------------------------------------------------------------------------*/
/* Waiter Class definition                                                  */
/*--------------------------------------------------------------------------*/
::CLASS Waiter PUBLIC                  

/*--------------------------------------------------------------------------*/
::METHOD wait UNGUARDED                /* This method allows to pause       */ 
  use arg seconds                      
                                       /* Create a 'complete' message       */  
  msg = .message~new(self, "COMPLETE", 'A', .array~new)
  .alarm~new(seconds, msg)             /* Wait and then send the message    */ 
  rc = msg~result                      /* Block on message completion       */ 

/*--------------------------------------------------------------------------*/
::METHOD complete UNGUARDED
  return 0                             /* Must have a return with a value!  */ 
