/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*    Kurt Maerker       Kurt_Maerker@vnet.ibm.com                          */
/*                                                              03/03/97    */
/*  Sockets.frm  -  Classes for using TCP Sockets                           */
/*                                                                          */
/*    Requirements:                                                         */
/*       RXSOCK library (Rexx interface to TCP Sockets library)             */
/*                                                                          */
/*--------------------------------------------------------------------------*/

                                       /* Load RxSock interface             */
if RxFuncQuery("SockLoadFuncs") then do
  rc = RxFuncAdd("SockLoadFuncs", "rxsock", "SockLoadFuncs")
  rc = "SockLoadFuncs"()
end

/*--------------------------------------------------------------------------*/
/* tcpSocket Class definition                                               */
/*--------------------------------------------------------------------------*/
::CLASS tcpSocket public

/*--------------------------------------------------------------------------*/
::METHOD init
  expose descriptor notClosed

  notClosed = 1
  if arg() = 1 then                    /* Socket exists; accept descriptor  */
    descriptor = arg(1)                /* Simply reuse this socket          */
  else do
    domain = "AF_INET"                 /* "AF_INET" is default domain       */
    type = "SOCK_STREAM"               /* "SOCK_STREAM" is default type     */
    protocol = 0                       /* use the default protocol          */
                                       /* Create a specific socket          */
    descriptor = SockSocket(domain, type, protocol)
    if descriptor = -1 then do         /* Failed to create socket           */
      say "Error on SockSocket:" errno
      return -1
    end 
    else do                            /* Socket successfully created       */  
      say self~string "created."
      return 0
    end
  end

/*--------------------------------------------------------------------------*/
::METHOD string
  expose descriptor

  return "tcpSocket" descriptor        /* Return the default string         */

/*--------------------------------------------------------------------------*/
::METHOD accept                        /* Accept client request             */
  expose descriptor

  connection = SockAccept(descriptor)  /* ... this is a blocking call!      */
  if rc = -1 then do
    say "Error on SockAccept:" errno
    return -1
  end
  return .tcpSocket~new(connection)    /* Return a new client socket        */ 

/*--------------------------------------------------------------------------*/
::METHOD bind                          /* Bind local name (port) to socket  */
  expose descriptor
  use arg port

  server.!family = "AF_INET"           /* Default domain                    */ 
  server.!port = port                  /* Local name (port)                 */ 
  server.!addr = "INADDR_ANY"          /* Accept any client address         */

  rc = SockBind(descriptor, "server.!")
  if rc = -1 then do
    say "Error on SockBind:" errno
    return -1
  end
  else 
    return 0

/*--------------------------------------------------------------------------*/
::METHOD connect                       /* Connect client socket to server   */
  expose descriptor
  use arg port, addr

  server.!family = "AF_INET"           /* Use default server domain         */ 
  server.!port = port                  /* at wellknown port                 */                     
  server.!addr = addr                  /* with IP address                   */

  rc = SockConnect(descriptor, "server.!")
  if rc = -1 then do
    say "Error on SockConnect:" errno
    return -1
  end
  return 0

/*--------------------------------------------------------------------------*/
::METHOD listen                        /* Listen to client's conn. request  */
  expose descriptor                    /* queue with max. backlog requests  */ 
  use arg backlog                      

  rc = SockListen(descriptor, backlog)
  if rc = -1 then do
    say "Error on SockListen:" errno
    return -1
  end
  return 0
                     
/*--------------------------------------------------------------------------*/
::METHOD DotAddress                    /* Get dotaddr. of system on socket  */
  expose descriptor

  rc = SockGetPeerName(descriptor, "Addr.!")
  if rc = -1 then return -1
  return Addr.!addr

/*--------------------------------------------------------------------------*/
::METHOD HostName                      /* Get hostname of system on socket  */ 
  expose descriptor

  DotAddr = self~DotAddress            /* Retrieve dot-address of host      */ 
  if DotAddr = -1 then  
    return -1                          /* No host available, socket invalid */ 
  else do                              /* Retrieve host name from address   */
    rc = SockGetHostByAddr(DotAddr, "Host.!")
    if rc = -1 then                      
      return "*Unknown*"               /* Host name not available           */
    else 
      return Host.!name                /* Return the host name              */
  end

/*--------------------------------------------------------------------------*/
::METHOD SendData UNGUARDED            /* Send data over socket             */ 
  expose descriptor
  use arg data
 
  rc = SockSend(descriptor, data)
  return rc

/*esc = x2c('00')                      /* use null for separator            */ 
  header = 'size=' data~length         /* store length in header            */
  data = header || esc || data         /* prefix data by header             */
  do while data \= ''
    parse var data packet 2000 data    /* 2000 bytes per packet             */
    rc = SockSend(descriptor, packet)  /* send the packet                   */
  end
  return 0                                                                  */

/*--------------------------------------------------------------------------*/
::METHOD ReceiveData UNGUARDED         /* Receive data from socket          */  
  expose descriptor                    /* ... this is a blocking read!      */ 
  
  rc = SockRecv(descriptor, "packet", 2000)
  if rc > 0 then 
    return packet                      /* Return the received packet of data*/ 
  else do
    self~shutdown                      /* Shutdown communication on socket  */ 
    self~close                         /* and close it                      */
    return rc
  end

/*esc = x2c('00')                      /* use null for separator            */ 
  rc = SockRecv(descriptor, "data", 2000)/* receive first packet            */
  parse var data 'size=' size (esc) data /* parse the header                */
  do while data~length < size
    rc = SockRecv(descriptor, "packet", 2000)/* receive next packet         */
    data = data || packet              /* append packet to data             */
  end
  return data                                                               */

/*--------------------------------------------------------------------------*/
::METHOD Shutdown UNGUARDED            /* Shutdown communication on socket  */ 
  expose descriptor notClosed 
  use arg mode
    
  if arg() = 0 then mode = 2           /* default: shutdown for read/write  */ 

  rc = SockShutDown(descriptor, mode)
  if rc \= 0 then do
    say "Error on SockShutDown:" errno
    return rc
  end
   
  say self~string "shutdown"
  return 0

/*--------------------------------------------------------------------------*/
::METHOD Close                         /* Recycle the socket resource       */   
  expose descriptor notClosed

  rc = SockSoClose(descriptor)
  if rc = -1 then do
    say "Error on SockSoClose:" errno  /* Problem with closing socket       */
    return -1
  end
  say self~string "closed"
  notClosed = 0                        /* Set its status closed             */ 
  return 0

/*--------------------------------------------------------------------------*/
::METHOD StillOpen UNGUARDED           /* Provide status information        */
  expose NotClosed

  return NotClosed

/*--------------------------------------------------------------------------*/
::METHOD uninit                        /* Just in case we forgot the Close  */  
  expose notClosed

  if notClosed then self~Close


/*--------------------------------------------------------------------------*/
/* tcpHost Class definition                                                 */
/*                                                                          */
/*   tcpHost objects are used to test a supplied hostname. If the host is   */
/*   found, information about it is available via various methods.          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
::CLASS tcpHost PUBLIC

/*--------------------------------------------------------------------------*/
::METHOD init
  expose name address addressArray aliasArray
  use arg hoststring
  
  if arg() = 0 | hoststring = '' then
    hoststring = SockGetHostId()

  if hoststring~left(1)~datatype('N') then
    rc = SockGetHostByAddr(hoststring, "Host.!")
  else
    rc = SockGetHostByName(hoststring, "Host.!")

  if rc = 0 then do
    name = ""
    address = ""
    say "Error" h_errno "accessing host. '" || hoststring || "' unknown."
    return 
  end 

  name = Host.!name
  address = Host.!addr

  addressArray = .array~new
  do i = 1 to Host.!addr.0
    addressArray[i] = Host.!addr.i
  end
    
  aliasArray = .array~new
  do i = 1 to Host.!alias.0
    aliasArray[i] = Host.!alias.i
  end

/*--------------------------------------------------------------------------*/
::METHOD name
  expose name 
  return name

/*--------------------------------------------------------------------------*/
::METHOD address
  expose address 
  return address

/*--------------------------------------------------------------------------*/
::METHOD addressArray
  expose addressArray
  return addressArray~copy

/*--------------------------------------------------------------------------*/
::METHOD aliasArray
  expose aliasArray
  return aliasArray~copy
