/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*    Kurt Maerker       Kurt_Maerker@vnet.ibm.com                          */
/*                                                              03/03/97    */
/*  Clients.frm  -  Classes for TCP/IP Clients                              */
/*                                                                          */
/*    Client program for the mirror server. Ask the user for a string,      */ 
/*    send the string to the mirror server and print the mirrored string.   */
/*                                                                          */
/*    Parameters:                                                           */
/*      Port:   server port number                                          */
/*      Server: alias name of server                                        */
/*                                                                          */
/*--------------------------------------------------------------------------*/

::REQUIRES "sockets.frm"               /* Load the sockets framework        */   

/*--------------------------------------------------------------------------*/
/* tcpClient Class definition                                               */
/*--------------------------------------------------------------------------*/
::CLASS tcpClient PUBLIC

/*--------------------------------------------------------------------------*/
::METHOD init                          /* Get sockets and binds port        */ 
  expose socket port connected         /* Attributes of a tcpClient object  */
  use arg server, port                 /* Arguments are server and port     */        
         
  socket = .tcpSocket~new              /* Get a stream socket               */
  host = .tcpHost~new(server)          /* Create a host instance            */
  connected = 0                        /* Initialize connect status         */ 

  address = host~address               /* Get server address from host      */
  if address \= "" then do             /* Valid address obtained?           */
    rc = socket~connect(port, address) /* Connect port of server            */
    connected = (rc = 0)               /* Set connect status                */
  end
  server = host~Name                   /* Get the server name               */
  if connected then 
    say "Client connected to server" server "at port" port
  else 
    say "Client could not establish connection to server" server "at port" port   
  return

/*--------------------------------------------------------------------------*/
::METHOD connected                     /* Show connection status            */   
  expose connected
  return connected

/*--------------------------------------------------------------------------*/
::METHOD send UNGUARDED                /* Send message to server            */
  expose socket port
  use arg message
  
  return socket~SendData(message)

/*--------------------------------------------------------------------------*/
::METHOD receive UNGUARDED             /* Receive return message from server*/
  expose socket port output
  
  output = socket~ReceiveData
  if socket~stillOpen then             /* Socket still open?                */          
    rc = output~length
  else do                              /* Received data ok, return # bytes  */
    rc = output
    output = ""                        /* Indicate error                    */ 
  end
  return rc

/*--------------------------------------------------------------------------*/
::METHOD promptInput                   /* prompt input from user            */
  expose socket port input
  use arg prompt
  
  say ""
  say prompt                           /* display prompt message            */

  signal on syntax                     /* Intercept ctrl-c interrupts       */
                                       
  do until Input \= ""                 /* read input from STDIN             */
    Parse Pull Input
  end

  rc = socket~SendData(Input)          /* send input string to server       */
  if rc < 0 then                     
    say "Lost connection to server on port" port  
  return rc

syntax:                                /* process when ctrl-c was pressed   */
  say "Client interrupted by user"
  return -1

/*--------------------------------------------------------------------------*/
::METHOD Input                         /* provide input string              */
  expose socket input
  return input

/*--------------------------------------------------------------------------*/
::METHOD Output UNGUARDED              /* provide output string             */
  expose socket output
  return output

/*--------------------------------------------------------------------------*/
::METHOD shutdown                      /* Shut down client                  */ 
  expose socket port connected

  if connected then do
    say ""
    say "Client disconnected from server" socket~HostName "at port" port
  end
  socket~Shutdown                      /* Shutdown the socket               */
  socket~Close                         /* Return the socket resource        */
  connected = 0                        /* Set socket status unconnected     */ 
  
