{*******************************************************}
{                     PHP4Delphi                        }
{               PHP - Delphi interface                  }
{                       v.5.0                           }
{ Author:                                               }
{ Serhiy Perevoznyk                                     }
{ serge_perevoznyk@hotmail.com                          }
{ http://users.chello.be/ws36637                        }
{*******************************************************}
{$I PHP.INC}
unit phpFunctions;

interface
 uses
   Windows, SysUtils, Classes, {$IFDEF VERSION6} Variants,
   {$ENDIF} ZendAPI, PHPAPI ;

type
  TParamType = (tpString, tpInteger, tpFloat, tpBoolean);

  TFunctionParam = class(TCollectionItem)
  private
    FName  : string;
    FValue : Variant;
    FParamType : TParamType;
  public
    property Value : variant read FValue write FValue;
    function GetDisplayName: string; override;
    procedure SetDisplayName(const Value: string); override;
    procedure AssignTo(Dest: TPersistent); override;
  published
    property Name : string read FName write SetDisplayName;
    property ParamType : TParamType read FParamType write FParamType;
  end;

  TFunctionParams = class(TCollection)
  private
    FOwner: TPersistent;
    function  GetItem(Index: Integer): TFunctionParam;
    procedure SetItem(Index: Integer; Value: TFunctionParam);
  protected
    function GetOwner: TPersistent; override;
    procedure SetItemName(Item: TCollectionItem); override;
  public
    constructor Create(Owner: TPersistent; ItemClass: TCollectionItemClass);
    function ParamByName(AName : string) : TFunctionParam;
    function Values(AName : string) : Variant;
    property Items[Index: Integer]: TFunctionParam read GetItem write SetItem; default;
  end;


  TPHPExecute = procedure(Sender : TObject; Parameters : TFunctionParams ; var ReturnValue : Variant;
                          ThisPtr : pzval;  TSRMLS_DC : pointer) of object;

  TPHPFunction = class(TCollectionItem)
  private
    FOnExecute : TPHPExecute;
    FFunctionName  : string;
    FTag       : integer;
    FFunctionParams: TFunctionParams;
    procedure SetFunctionParams(const Value: TFunctionParams);
  public
    ReturnValue : variant;
    constructor Create(Collection : TCollection); override;
    destructor Destroy; override;
    function  GetDisplayName: string; override;
    procedure SetDisplayName(const Value: string); override;
    procedure AssignTo(Dest: TPersistent); override;
  published
    property FunctionName : string read FFunctionName write SetDisplayName;
    property Tag  : integer read FTag write FTag;
    property Parameters: TFunctionParams read FFunctionParams write SetFunctionParams;
    property OnExecute : TPHPExecute read FOnExecute write FOnExecute;
  end;

  TPHPFunctions = class(TCollection)
  private
    FOwner: TPersistent;
    function GetItem(Index: Integer): TPHPFunction;
    procedure SetItem(Index: Integer; Value: TPHPFunction);
  protected
    function GetOwner: TPersistent; override;
    procedure SetItemName(Item: TCollectionItem); override;
  public
    constructor Create(Owner: TPersistent; ItemClass: TCollectionItemClass);
    property Items[Index: Integer]: TPHPFunction read GetItem write SetItem; default;
  end;


function IsParamTypeCorrect(AParamType :  TParamType; z : Pzval) : boolean;

implementation

function IsParamTypeCorrect(AParamType :  TParamType; z : Pzval) : boolean;
var
  ZType : integer;
begin
  ZType := Z^._type;
  case AParamType Of
   tpString  : Result := (ztype in [IS_STRING, IS_NULL]);
   tpInteger : Result := (ztype in [IS_LONG, IS_BOOL, IS_NULL]);
   tpFloat   : Result := (ztype in [IS_NULL, IS_DOUBLE, IS_LONG]);
   tpBoolean : Result := (ztype in [IS_NULL, IS_BOOL]);
    else
     Result := False;
   end;
end;

{ TPHPFunctions }

constructor TPHPFunctions.Create(Owner: TPersistent; ItemClass: TCollectionItemClass);
begin
  inherited Create(ItemClass);
  FOwner := Owner;
end;

function TPHPFunctions.GetItem(Index: Integer): TPHPFunction;
begin
  Result := TPHPFunction(inherited GetItem(Index));
end;

function TPHPFunctions.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

procedure TPHPFunctions.SetItem(Index: Integer; Value: TPHPFunction);
begin
  inherited SetItem(Index, TCollectionItem(Value));
end;


procedure TPHPFunctions.SetItemName(Item: TCollectionItem);
var
  I, J: Integer;
  ItemName: string;
  CurItem: TPHPFunction;
begin
  J := 1;
  while True do
  begin
    ItemName := Format('PHPFunction%d', [J]);
    I := 0;
    while I < Count do
    begin
      CurItem := Items[I] as TPHPFunction;
      if (CurItem <> Item) and (CompareText(CurItem.FunctionName, ItemName) = 0) then
      begin
        Inc(J);
        Break;
      end;
      Inc(I);
    end;
    if I >= Count then
    begin
      (Item as TPHPFunction).FunctionName := ItemName;
      Break;
    end;
  end;
end;

{ TPHPFunction }

procedure TPHPFunction.AssignTo(Dest: TPersistent);
begin
  if Dest is TPHPFunction then
  begin
    if Assigned(Collection) then Collection.BeginUpdate;
    try
      with TPHPFunction(Dest) do
      begin
        Tag := Self.Tag;
      end;
    finally
      if Assigned(Collection) then Collection.EndUpdate;
    end;
  end else inherited AssignTo(Dest);
end;

constructor TPHPFunction.Create(Collection: TCollection);
begin
  inherited;
  FFunctionParams := TFunctionParams.Create(TPHPFunctions(Self.Collection).GetOwner, TFunctionParam);
end;

destructor TPHPFunction.Destroy;
begin
  FFunctionParams.Free;
  inherited;
end;

function TPHPFunction.GetDisplayName: string;
begin
  if FFunctionName = '' then
   result :=  inherited GetDisplayName else
     Result := FFunctionName;
end;


procedure TPHPFunction.SetDisplayName(const Value: string);
var
  I: Integer;
  F: TPHPFunction;
begin
  if AnsiCompareText(Value, FFunctionName) <> 0 then
  begin
    if Collection <> nil then
      for I := 0 to Collection.Count - 1 do
      begin
        F := TPHPFunctions(Collection).Items[I];
        if (F <> Self) and (F is TPHPFunction) and
          (AnsiCompareText(Value, F.FunctionName) = 0) then
          raise Exception.Create('Duplicate function name');
      end;
    FFunctionName := Value;
    Changed(False);
  end;
end;


procedure TPHPFunction.SetFunctionParams(const Value: TFunctionParams);
begin
  FFunctionParams.Assign(Value);
end;


{ TFunctionParams }

constructor TFunctionParams.Create(Owner: TPersistent;
  ItemClass: TCollectionItemClass);
begin
  inherited Create(ItemClass);
  FOwner := Owner;
end;

function TFunctionParams.GetItem(Index: Integer): TFunctionParam;
begin
  Result := TFunctionParam(inherited GetItem(Index));
end;

function TFunctionParams.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

function TFunctionParams.ParamByName(AName: string): TFunctionParam;
var
 i : integer;
begin
  Result := nil;
  for i := 0 to Count - 1 do
   begin
     if SameText(AName, Items[i].Name) then
      begin
        Result := Items[i];
        Break;
      end;
   end;
end;

procedure TFunctionParams.SetItem(Index: Integer; Value: TFunctionParam);
begin
  inherited SetItem(Index, TCollectionItem(Value));
end;

procedure TFunctionParams.SetItemName(Item: TCollectionItem);
var
  I, J: Integer;
  ItemName: string;
  CurItem: TFunctionParam;
begin
  J := 1;
  while True do
  begin
    ItemName := Format('Param%d', [J]);
    I := 0;
    while I < Count do
    begin
      CurItem := Items[I] as TFunctionParam;
      if (CurItem <> Item) and (CompareText(CurItem.Name, ItemName) = 0) then
      begin
        Inc(J);
        Break;
      end;
      Inc(I);
    end;
    if I >= Count then
    begin
      (Item as TFunctionParam).Name := ItemName;
      Break;
    end;
  end;
end;

function TFunctionParams.Values(AName: string): Variant;
var
 P : TFunctionParam;
begin
  Result := NULL;
  P := ParamByName(AName);
  if Assigned(P) then
   Result := P.Value;
end;

{ TFunctionParam }

procedure TFunctionParam.AssignTo(Dest: TPersistent);
begin
  if Dest is TFunctionParam then
  begin
    if Assigned(Collection) then Collection.BeginUpdate;
    try
      with TFunctionParam(Dest) do
      begin
        ParamType := Self.ParamType;
      end;
    finally
      if Assigned(Collection) then Collection.EndUpdate;
    end;
  end else inherited AssignTo(Dest);
end;

function TFunctionParam.GetDisplayName: string;
begin
  if FName = '' then
   result :=  inherited GetDisplayName else
     Result := FName;
end;

procedure TFunctionParam.SetDisplayName(const Value: string);
var
  I: Integer;
  F: TFunctionParam;
begin
  if AnsiCompareText(Value, FName) <> 0 then
  begin
    if Collection <> nil then
      for I := 0 to Collection.Count - 1 do
      begin
        F := TFunctionParams(Collection).Items[I];
        if ((F <> Self) and (F is TFunctionParam) and
          (AnsiCompareText(Value, F.Name) = 0)) then
          raise Exception.Create('Duplicate parameter name');
      end;
    FName := Value;
    Changed(False);
  end;
end;


end.