{ OBTLIB.PAS : Ordered Binary Tree object library

  Title    : OBTLIB
  Version  : 2.2
  Date     : Nov 11,1996
  Author   : J R Ferguson
  Language : Turbo Pascal v7.0
  Usage    : Unit
}

UNIT ObtLib;

INTERFACE
Uses ObjLib;

type

  P_Obt_Node       = ^T_Obt_Node;
  P_Obt_Tree       = ^T_Obt_Tree;
  P_Obt_SearchTree = ^T_Obt_SearchTree;

  T_Obt_Node = record
  { Node information record, used in T_Obt_Tree and T_Obt_SearchTree. }
    Item           : Pointer;
    Prev, Next     : P_Obt_Node;
  end;

  T_Obt_Tree    = object(T_Object)
  { Unbalanced Binary Tree. The nodes need not have unique keys.
    There are no methods to find or process individual nodes, once they
    are inserted.
  }
    Root        : P_Obt_Node;
    CompareFunc : T_CompareFunc;
    constructor Init(V_CompareFunc: T_CompareFunc);
    { Create new (empty) tree, setting the compare function on which to
      base its node ordering.
      Override: sometimes.
    }
    destructor  Done; virtual;
    { Discard tree after deleting and disposing of all items in it by
      calling DeleteAll.
      Override: sometimes.
    }
    procedure   FreeItem(V_Item: Pointer); virtual;
    { Disposes of the specified item, freeing any memory allocated by it.
      The default FreeItem does nothing. In most cases you will override
      this mehod. When V_Item is an object pointer of type P_Item, the
      code could be:
        Dispose(P_Item(V_Item),Done);
      FreeItem is called by DeleteAll for each node item that is not nil,
      before that node itself is deallocated. It should never be called
      directly, because that would leave a node with an invalid Item
      pointer in the tree.
      Override: often.
    }
    procedure   DeleteAll; virtual;
    { Delete and dispose of all nodes in the tree. Before a node is
      deleted, FreeItem is called with its Item pointer as a parameter,
      unless the Item pointer is nil.
      The result is an empty tree.
      Override: seldom.
    }
    function    Empty: boolean; virtual;
    { Test if the tree is empty
      Override: never.
    }
    function    Insert(V_Item: Pointer): boolean; virtual;
    { Insert a node containing the specified item into the tree.
      The function result is true on success, and false if there is not
      enough memory to insert the item, provided that a HeapError function
      has been installed in such a way that New and Getmem return nil on
      failure (see Turbo Pascal manual).
      If no HeapError function is installed, a memory overflow will result
      in a runtime error.
      Override: never.
    }
    procedure   InOrder(V_Process: T_ProcessProc); virtual;
    { InOrder tree traversal, processing all node items with the
      specified procedure in the order defined by CompareFunc.
      Override: never.
    }
    procedure   PreOrder(V_Process: T_ProcessProc); virtual;
    { PreOrder tree traversal, processing all node items with the
      specified procedure.
      The nodes are processed after their child subtrees have been
      processed (in the order of the predecessor subtree followed by
      the sucessor subtree, as defined by CompareFunc).
      Override: never.
    }
    procedure   PostOrder(V_Process: T_ProcessProc); virtual;
    { PostOrder tree traversal, processing all node items with the
      specified procedure.
      The nodes are processed before their child subtrees are
      processed (in the order of the predecessor subtree followed by
      the sucessor subtree, as defined by CompareFunc).
      Override: never.
    }
  end;


  T_Obt_SearchTree = object(T_Obt_Tree)
  { Unbalanced Binary Search Tree. This is a descendant of T_Obt_Tree which
    maintains the following constraints:
     - All nodes in the Prev subtree of each node precede that node.
     - All nodes in the Next subtree of each node follow that node.
    As a consequence, all tree nodes must have unique keys.
    In addition to the methods provided by T_Obt_Tree, this object also
    has a Search method to find an individual node item.
  }
    function    Insert(V_Item: Pointer): boolean; virtual;
    { Insert an item into the tree, in such a way that the binary search
      tree constraint is maintained.
      The result is true if the instertion succeeded, and false if the
      key of V_Item was already present in the tree.
      By installing a HeapError function the function can also be made to
      return false on memory overflow as explained with T_ObtTree.Insert.
      Override: never.
    }
    function    Search(V_Item: Pointer): Pointer; virtual;
    { Search for a tree node with an item that compares equal to V_Item.
      If found, return a pointer to its item. Otherwise, return nil.
      Override: never.
    }
    function    Delete(V_Item: Pointer): boolean; virtual;
    { Search for a tree node with an item that compares equal to V_Item.
      If found, delete that node from the tree and dispose of it, after
      calling FreeItem(V_Item) if V_Item <> nil.
      Returns success (false if the item was not found in the tree, true
      if it was).
      Override: never.
    }
  private
    function    _Locate(V_Item:Pointer; var V_Node:P_Obt_Node): integer;
    { Search for V_Item and return the result of the last CompareFunc call.
      Set V_Node to point to the node containing V_Item's key, or to
      the node where V_Item is to be inserted if it was not found. }
  end;


IMPLEMENTATION


{ --- T_Obt_Tree methods --- }

constructor T_Obt_Tree.Init(V_CompareFunc: T_CompareFunc);
begin inherited Init; Root:= nil; CompareFunc:= V_CompareFunc; end;

destructor  T_Obt_Tree.Done;
begin DeleteAll; inherited Done; end;

procedure   T_Obt_Tree.FreeItem(V_Item: Pointer);
begin end;

procedure   T_Obt_Tree.DeleteAll;
var Item: Pointer;
  procedure discard(node: P_Obt_Node);
  begin if node<>nil then begin
    discard(node^.Prev); discard(node^.Next);
    if node^.Item <> nil then FreeItem(node^.Item);
    dispose(node);
  end end;
begin discard(Root); Root:= nil; end;

function    T_Obt_Tree.Empty: boolean;
begin Empty:= Root=nil end;

function    T_Obt_Tree.Insert(V_Item: Pointer): boolean;
  function SubTreeInsert(var node: P_Obt_Node): boolean;
  begin
    if node=nil then begin
      new(node);
      if node = nil then Insert:= false
      else begin
        node^.Prev:= nil; node^.Next:= nil; node^.Item:= V_Item;
        Insert:= true;
      end;
    end
    else begin
      if CompareFunc(V_Item,node^.Item) < 0
        then Insert:= SubTreeInsert(node^.Prev)
        else Insert:= SubTreeInsert(node^.Next);
    end;
  end;
begin { T_Obt_Tree.Insert }
  Insert:= SubTreeInsert(Root);
end;

procedure   T_Obt_Tree.InOrder(V_Process: T_ProcessProc);
  procedure traverse(node: P_Obt_Node);
  begin if node<>nil then begin
    traverse(node^.Prev); V_Process(node^.Item); traverse(node^.Next);
  end end;
begin if @V_Process <> nil then traverse(Root); end;

procedure   T_Obt_Tree.PreOrder(V_Process: T_ProcessProc);
  procedure traverse(node: P_Obt_Node);
  begin if node<>nil then begin
    V_Process(node^.Item); traverse(node^.Prev); traverse(node^.Next);
  end end;
begin if @V_Process <> nil then traverse(Root); end;

procedure   T_Obt_Tree.PostOrder(V_Process: T_ProcessProc);
  procedure traverse(node: P_Obt_Node);
  begin if node<>nil then begin
    traverse(node^.Prev); traverse(node^.Next); V_Process(node^.Item);
  end end;
begin if @V_Process <> nil then traverse(Root); end;


{ --- T_Obt_SearchTree methods --- }

function    T_Obt_SearchTree.Insert(V_Item: Pointer): boolean;
var cmp: integer; node, OldNode: P_Obt_Node;
begin
  cmp:= _Locate(V_Item,OldNode);
  if cmp = 0 then Insert:= false
  else begin
    new(node);
    if node=nil then Insert:= false
    else begin
      node^.Item:= V_Item; node^.Prev:= nil; node^.Next:= nil;
      if Root = nil   then Root:= node
      else if cmp < 0 then OldNode^.Prev:= node
      else                 OldNode^.Next:= node;
      Insert:= true;
    end;
  end;
end;

function    T_Obt_SearchTree.Search(V_Item: Pointer): Pointer;
var node: P_Obt_Node;
begin
  if _Locate(V_Item,node) = 0 then Search:= node^.Item else Search:= nil;
end;


function    T_Obt_SearchTree.Delete(V_Item: Pointer): boolean;
var DelNode: P_Obt_Node;

  procedure subdel(var node: P_Obt_Node);
  begin
    if node^.Next <> nil then subdel(node^.Next)
    else begin
      DelNode^.Item:= node^.Item;
      DelNode:= node; node:= node^.Prev;
    end;
  end;

  function del(var node: P_Obt_Node): boolean;
  var cmp: integer;
  begin
    if node = nil then del:= false
    else begin
      cmp:= CompareFunc(V_Item,node^.Item);
      if      cmp < 0 then del:= del(node^.Prev)
      else if cmp > 0 then del:= del(node^.Next)
      else begin
        if node^.Item<>nil then FreeItem(node^.Item);
        DelNode:= node;
        if      node^.Next = nil then node:= node^.Prev
        else if node^.Prev = nil then node:= node^.Next
        else subdel(DelNode^.Prev);
        dispose(DelNode); del:= true;
      end;
    end;
  end;

begin {T_Obt_SearchTree.delete}
  Delete:= del(Root);
end;


function    T_Obt_SearchTree._Locate(V_Item: Pointer;
                       var V_Node: P_Obt_Node): integer;
var node: P_Obt_Node; result: integer;
begin
  result:= -1; node:= root; V_Node:= nil;
  while (result <> 0) and (node <> nil) do begin
    V_Node:= node;
    result:= CompareFunc(V_Item,node^.Item);
    if      result < 0 then node:= node^.Prev
    else if result > 0 then node:= node^.Next;
  end;
  _Locate:= result;
end;


END.
