//===================================================================
// correlat.hpp
//
// Version 1.1
//
// Written by:
//   Brent Worden
//   WordenWare
//   email:  Brent@Worden.org
//
// Copyright (c) 1998-1999 WordenWare
// 
// Created:  August 28, 1998
// Revised:  April 10, 1999
//===================================================================

#ifndef _CORRELAT_HPP_
#define _CORRELAT_HPP_

#include "descript.hpp"
#include "ranking.hpp"
#include "vector.hpp"

NUM_BEGIN

template<class Iter1, class Iter2>
inline double covariance(Iter1 first1, Iter1 last1, Iter2 first2)
//-------------------------------------------------------------------
// Returns the covariance of the corresponding elements in
// [first1, last1) and [first2, first2 + (last1 - first1) ).
//-------------------------------------------------------------------
{
    return sumproduct(first1, last1, first2) / length(first1, last1);
}

template<class Iter1, class Iter2>
inline double correlation(Iter1 first1, Iter1 last1, Iter2 first2)
//-------------------------------------------------------------------
// Returns the Pearson's product moment correlation coefficient of
// the corresponding elements in [first1, last1) and [first2,
// first2 + (last1 - first1) ).
//-------------------------------------------------------------------
{
    return sumproduct(first1, last1, first2) / sqrt(sumsquare(first1, last1) *
		sumsquare(first2, first2 + (last1 - first1)));
};

template<class Iter1, class Iter2>
double kendall(Iter1 first1, Iter1 last1, Iter2 first2)
//-------------------------------------------------------------------
// Returns Kenall's Tau correlation coefficient of the corresponding
// values in [first1, last1) and [first2, first2 + (last1 - first1)).
//-------------------------------------------------------------------
{
    int nc, nd, ey, ex;
    int i, j;
    int n = length(first1, last1);
    
    nc = nd = ey = ex = 0;
    
    for(i = 0; i < n; ++i){
        for(j = 0; j < i; ++j){
            if(*(first1 + i) < *(first1 + j)){
                if(*(first2 + i) < *(first2 + j)){
                    ++nc;
                } else if(*(first2 + i) > *(first2 + j)){
                    ++nd;
                } else {
                    ++ex;
                }
            } else if(*(first1 + i) > *(first1 + j)){
                if(*(first2 + i) < *(first2 + j)){
                    ++nd;
                } else if(*(first2 + i) > *(first2 + j)){
                    ++nc;
                } else {
                    ++ex;
                }
            } else {
                ++ey;
                if(*(first2 + i) == *(first2 + j)){
                    ++ex;
                }
            }
        }
    }
    
    return (double)(nc - nd) / sqrt((double)((nc+nd+ey)*(nc+nd+ex)));
};

template<class Iter1, class Iter2>
double spearman(Iter1 first1, Iter1 last1, Iter2 first2)
//-------------------------------------------------------------------
// Returns Spearman's correlation coefficient of the corresponding
// values in [first1, last1) and [first2, first2 + (last1 - first1)).
// Note:  Ties in the values are not accounted for.
//-------------------------------------------------------------------
{
    int n = length(first1, last1);
	Vector<double> r1(n);
	Vector<double> r2(n);
    
    rank(first1, last1, r1.begin());
    rank(first2, first2 + n, r2.begin());
    
    return correlation(r1.begin(), r1.end(), r2.begin());
};

NUM_END

#endif

//===================================================================
// Revision History
//
// Version 1.0 - 08/28/1998 - New.
// Version 1.1 - 04/10/1999 - Added Numerics namespace.
//                            Replaced std::vector with num::Vector.
//===================================================================

