#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <curses.h>
#include <unistd.h>
#include "cpu.h"
#include "soc.h"
#include "vUART.h"

#define VERBOSE				0
#define FIFO_SIZE			64

#define CONSOLE_WIDTH		80
#define CONSOLE_HEIGHT		25

#define VOFFSET				8
#define HOFFSET				2

//multi-byte fifo writes not implemented as we do not use them
//our chars are also always 8 bits and nevr have errors



enum UartState {
	UartStateDeselected,
	UartStateStartByteWait,
	UartStateRegRead,
	UartStateRegWrite,

	UartStatePostCommandIdle,
};

struct FIFO {
	uint8_t readPtr, writePtr, numBytesPresent;
	uint8_t buf[FIFO_SIZE];
};

struct VUART {
	struct VSPI *vspi;

	//regs
	uint8_t EFR, LCR, DLL, DLH, IER, IIR, MCR, LSR, MSR, TCR, SPR, regE, regF, FCR, TLR, XON1, XON2, XOFF1, XOFF2;

	//curses screen
	uint8_t screen[CONSOLE_HEIGHT][CONSOLE_WIDTH];
	uint16_t screenCol;


	struct FIFO txFifo, rxFifo;
	uint8_t curReg;
	enum UartState state;
};

static uint8_t uartPrvFifoUsed(const struct FIFO *fifo)
{
	return fifo->numBytesPresent;
}

static uint8_t uartPrvFifoRead(struct FIFO *fifo)
{
	uint8_t ret;

	if (!fifo->numBytesPresent) {
		fprintf(stderr, "READING EMPTY FIFO\n");
		abort();
	}
	ret = fifo->buf[fifo->readPtr];
	if (++fifo->readPtr == FIFO_SIZE)
		fifo->readPtr = 0;
	fifo->numBytesPresent--;

	return ret;
}

static void uartPrvFifoWrite(struct FIFO *fifo, uint8_t val)
{
	if (fifo->numBytesPresent == FIFO_SIZE) {
		fprintf(stderr, "WRITING FULL FIFO\n");
		abort();
	}
	fifo->buf[fifo->writePtr] = val;
	if (++fifo->writePtr == FIFO_SIZE)
		fifo->writePtr = 0;
	fifo->numBytesPresent++;
}

static uint8_t uartPrvGetCurFifoSize(struct VUART *uart)
{
	return (uart->FCR & 1) ? FIFO_SIZE : 1;
}

static uint8_t vuartPrvRegRead(struct VUART *uart, uint8_t reg)
{
	switch (reg) {
		case 0:			//RHR or DLL
			if (uart->LCR & 0x80)
				return uart->DLL;
			else
				return uartPrvFifoRead(&uart->rxFifo);

		case 1:			//IER or DLH
			if (uart->LCR & 0x80)
				return uart->DLH;
			else
				return uart->IER;

		case 2:			//IIR or EFR
			if (uart->LCR == 0xBF)
				return uart->EFR;
			else
				return uart->IIR;

		case 3:			//LCR
			return uart->LCR;

		case 4:			//MCR or XON1
			if (uart->LCR == 0xBF)
				return uart->XON1;
			else
				return uart->MCR;

		case 5:			//LSR or XON2
			if (uart->LCR == 0xBF)
				return uart->XON2;
			else
				return uart->LSR;

		case 6:			//MSR, TCR, or XOFF1
			if (uart->LCR == 0xBF)
				return uart->XOFF1;
			else if ((uart->MCR & 4) && (uart->EFR & 16))
				return uart->TCR;
			else
				return uart->MSR;

		case 7:			//SPR,TLR, or XOFF2
			if (uart->LCR == 0xBF)
				return uart->XOFF2;
			else if ((uart->MCR & 4) && (uart->EFR & 16))
				return uart->TLR;
			else
				return uart->SPR;

		case 8:			//TXLVL
			return 64 - uartPrvFifoUsed(&uart->txFifo);

		case 9:			//RXLVL
			return uartPrvFifoUsed(&uart->rxFifo);

		case 14:		//reg 0x0e
			return uart->regE;

		case 15:		//reg 0x0f
			return uart->regF;

		default:
			fprintf(stderr, "unknown reg read: [%u]\n", reg);
			abort();

	}
}

static void vuartPrvRegWrite(struct VUART *uart, uint8_t reg, uint8_t val)
{
	switch (reg) {
		case 0:			//RHR or DLL
			if (uart->LCR & 0x80)
				uart->DLL = val;
			else
				uartPrvFifoWrite(&uart->txFifo, val);
			break;

		case 1:			//IER or DLH
			if (uart->LCR & 0x80)
				uart->DLH = val;
			else
				uart->IER = val;
			break;

		case 2:			//FCR or EFR
			if (uart->LCR == 0xBF)
				uart->EFR = val;
			else
				uart->FCR = val & ~ 0x06;
			break;

		case 3:			//LCR
			uart->LCR = val;
			break;

		case 4:			//MCR or XON1
			if (uart->LCR == 0xBF)
				uart->XON1 = val;
			else
				uart->MCR = val;
			break;

		case 5:			//LSR or XON2
			if (uart->LCR == 0xBF)
				uart->XON2 = val;
			else
				goto unknown_reg;
			break;

		case 6:			//MSR, TCR, or XOFF1
			if (uart->LCR == 0xBF)
				uart->XOFF1 = val;
			else if ((uart->MCR & 4) && (uart->EFR & 16))
				uart->TCR = val;
			else
				goto unknown_reg;
			break;

		case 7:			//SPR,TLR, or XOFF2
			if (uart->LCR == 0xBF)
				uart->XOFF2 = val;
			else if ((uart->MCR & 4) && (uart->EFR & 16))
				uart->TLR = val;
			else
				uart->SPR = val;
			break;

		case 14:		//reg 0x0e
			uart->regE = val;
			break;

		case 15:		//reg 0x0f
			uart->regF = val;
			break;

	unknown_reg:
		default:
			fprintf(stderr, "unknown reg written: 0x%02x -> [%u]\n", val, reg);
			abort();

	}
}

static uint8_t vuartPrvSpiProvideByteF(void *userData)
{
	struct VUART *uart = (struct VUART *)userData;
	uint8_t ret = 0;

	switch (uart->state) {

		case UartStateRegRead:
			ret = vuartPrvRegRead(uart, uart->curReg);
			break;

		default:
			//we have nothing to say
			break;
	}

	if (VERBOSE)
		fprintf(stderr, "VUART: sent 0x%02x\n", ret);
	return ret;
}

static void vuartPrvSpiAcceptByteF(void *userData, uint8_t byte)
{
	struct VUART *uart = (struct VUART *)userData;

	if (VERBOSE)
		fprintf(stderr, "VUART: got  0x%02x\n", byte);

	switch (uart->state) {

		case UartStateStartByteWait:
			if (byte & 6) {
				fprintf(stderr, "VUART: CH must be zero, byte was 0x%02x\n", byte);
				abort();
			}
			uart->curReg = (byte >> 3) & 15;
			if (byte & 0x80)
				uart->state = UartStateRegRead;
			else
				uart->state = UartStateRegWrite;
			break;

		case UartStateRegWrite:
			vuartPrvRegWrite(uart, uart->curReg, byte);
			uart->state = UartStatePostCommandIdle;
			break;

		case UartStatePostCommandIdle:
			fprintf(stderr, "VUART: got data after the requisite bytes: 0x%02x\n", byte);
			abort();
			break;

		case UartStateRegRead:	//called after data is provided so this is the right place to do this
			uart->state = UartStatePostCommandIdle;
			break;

		case UartStateDeselected:
			//we do not care
			break;
	}
}

static void vuartPrvSpiSelectionChanged(void *userData, bool selected)
{
	struct VUART *uart = (struct VUART *)userData;

	if (selected) {

		if (VERBOSE)
			fprintf(stderr, "===== VUART :selected =====\n");

		switch (uart->state) {
			case UartStateDeselected:
				uart->state = UartStateStartByteWait;
				break;

			default:
				fprintf(stderr, "VUART: select in state %u\n", uart->state);
				abort();
		}
	}
	else {

		if (VERBOSE)
			fprintf(stderr, "==== VUART: deselected ====\n");

		switch (uart->state) {
			case UartStatePostCommandIdle:
				uart->state = UartStateDeselected;
				break;

			default:
				fprintf(stderr, "VUART: deselect in state %u\n", uart->state);
				abort();
		}
	}
}

struct VUART* vuartInit(void)
{
	struct VUART *uart = calloc(1, sizeof(struct VUART));

	if (uart) {
		uart->vspi = vspiInit("VUART", SpiMode0);
		
		if (uart->vspi) {
			
			uint_fast16_t i;

			uart->LCR = 0x1D;
			uart->IIR = 0x01;
			uart->LSR = 0x60;
			memset(uart->screen, ' ', sizeof(uart->screen));

			vspiDeviceRegister(uart->vspi, vuartPrvSpiProvideByteF, vuartPrvSpiAcceptByteF, vuartPrvSpiSelectionChanged, uart);

			//draw border
			for (i = 0; i < CONSOLE_WIDTH + 2; i++) {

				mvaddch(VOFFSET, i + HOFFSET, '_');
				mvaddch(CONSOLE_HEIGHT + 1 + VOFFSET, i + HOFFSET, ' -');
			}
			for (i = 0; i < CONSOLE_HEIGHT + 2; i++) {

				mvaddch(VOFFSET + i, HOFFSET, ']');
				mvaddch(VOFFSET + i, CONSOLE_WIDTH + 2 + HOFFSET, '[');
			}

			refresh();

			return uart;
		}
		free(uart);
		uart = NULL;
	}

	return uart;
}

struct VSPI* vuartGetVSPI(struct VUART *uart)
{
	return uart->vspi;
}

void vuartDeinit(struct VUART *uart)
{
	vspiDestroy(uart->vspi);
	free(uart);
}

static void vuartScroll(struct VUART *uart)
{
	uint_fast16_t r, c;

	//scroll
	memmove(uart->screen[0], uart->screen[1], sizeof(uart->screen) - sizeof(uart->screen[0]));
	memset(uart->screen[CONSOLE_HEIGHT - 1], ' ', sizeof(uart->screen[CONSOLE_HEIGHT - 1]));

	//redraw curses
	for (r = 0; r < CONSOLE_HEIGHT; r++) {

		for (c = 0; c < CONSOLE_WIDTH; c++) {

			mvaddch(r + VOFFSET + 1, c + HOFFSET + 1, uart->screen[r][c]);
		}
	}
}

void vuartPeriodic(struct VUART *uart)
{
	if (uartPrvFifoUsed(&uart->txFifo)) {
		uint8_t chr = uartPrvFifoRead(&uart->txFifo);

		if (chr == '\r')
			uart->screenCol = 0;
		else if (chr == 8) {	//backspace
			if (uart->screenCol) {

				uart->screenCol--;
				mvaddch(CONSOLE_HEIGHT - 1 + VOFFSET + 1, uart->screenCol + HOFFSET + 1, ' ');
				uart->screen[CONSOLE_HEIGHT - 1][uart->screenCol] = ' ';
			}
		}
		else {
			if (chr == '\n' || uart->screenCol == CONSOLE_WIDTH) {

				vuartScroll(uart);
				if (uart->screenCol == CONSOLE_WIDTH)	//match minicom with linewrap in behaviour
					uart->screenCol = 0;
			}
			
			if (chr != '\n') {
			
				mvaddch(CONSOLE_HEIGHT - 1 + VOFFSET + 1, uart->screenCol + HOFFSET + 1, chr);
				uart->screen[CONSOLE_HEIGHT - 1][uart->screenCol++] = chr;
			}
			refresh();
		}
	}
	//check for input if we have space (emulate flow control)
	if (uartPrvFifoUsed(&uart->rxFifo) < uartPrvGetCurFifoSize(uart)) {

		struct timeval limit = {};
		fd_set set;
		
		FD_ZERO(&set);
		FD_SET(0, &set);
		
		if (1 == select(1, &set, NULL, NULL, & limit)) {

			unsigned char ch;

			if (1 == read(0, &ch, 1)) {

				uartPrvFifoWrite(&uart->rxFifo, ch);
			}
		}
	}
}

bool vuartGetIrqPinVal(struct VUART *uart)
{
	return !uartPrvFifoUsed(&uart->rxFifo);
}







