#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "vSD.h"


#define VERBOSE		0					//can be 0, 1 or 2, or 3


#define INIT_COMMANDS_REQD					50
#define BUSY_BYTES_POST_WRITE				100

//very simple SD emulator for u4004 - does not support multiblock accesses or much else

#define R1_RSP_PARAM_ERROR			0x40
#define R1_RSP_ADDRESS_ERR			0x20
#define R1_RSP_ERZ_SEQ_ERR			0x10
#define R1_RSP_CRC_ERR				0x08
#define R1_RSP_ILLEGAL_CMD			0x04
#define R1_RSP_ERASE_RESET			0x02
#define R1_RSP_IN_IDLE_STATE		0x01

enum SdState {
	SdStateDeselected,
	SdStateCmdRx,
	SdStateSendR1,
	SdStateSendWait,
	SdStateSendData,
	SdStateDataRxWait,
	SdStateDataRx,
	SdStateDataRxedSendReply,
	SdStateBusy,
	SdStatePostCmdIdle,
};

struct VSD {
	VsdSecReadF readF;
	VsdSecWriteF writeF;
	void *userData;
	uint32_t numSec;

	struct VSPI *vspi;

	enum SdState state, nextState;
	uint8_t cmd[6];	//incl crc
	uint8_t data[515];
	bool isSDHC, isInited, crcOn, hadCmd8, acmd;
	uint32_t writeSecNo, numBytes, delayCounter, initCounter;
	union {
		struct {
			uint32_t cSize		:22;
		} hc;
		struct {
			uint32_t cSize		:12;
			uint32_t cSizeMult	:3;
			uint32_t readBlLen	:4;
		} sc;
	} size;
};

//typedef bool (*VsdSecReadF)(void *userData, uint32_t sec, uint8_t *dst);
//typedef bool (*VsdSecWriteF)(void *userData, uint32_t sec, const uint8_t *dst);

static uint8_t vsdPrvCrcAccount(uint8_t curCrc, uint8_t byte)
{
        static const uint8_t crcTab7[] = {              //generated from the iterative func :)
                0x00, 0x09, 0x12, 0x1b, 0x24, 0x2d, 0x36, 0x3f, 0x48, 0x41, 0x5a, 0x53, 0x6c, 0x65, 0x7e, 0x77,
                0x19, 0x10, 0x0b, 0x02, 0x3d, 0x34, 0x2f, 0x26, 0x51, 0x58, 0x43, 0x4a, 0x75, 0x7c, 0x67, 0x6e,
                0x32, 0x3b, 0x20, 0x29, 0x16, 0x1f, 0x04, 0x0d, 0x7a, 0x73, 0x68, 0x61, 0x5e, 0x57, 0x4c, 0x45,
                0x2b, 0x22, 0x39, 0x30, 0x0f, 0x06, 0x1d, 0x14, 0x63, 0x6a, 0x71, 0x78, 0x47, 0x4e, 0x55, 0x5c,
                0x64, 0x6d, 0x76, 0x7f, 0x40, 0x49, 0x52, 0x5b, 0x2c, 0x25, 0x3e, 0x37, 0x08, 0x01, 0x1a, 0x13,
                0x7d, 0x74, 0x6f, 0x66, 0x59, 0x50, 0x4b, 0x42, 0x35, 0x3c, 0x27, 0x2e, 0x11, 0x18, 0x03, 0x0a,
                0x56, 0x5f, 0x44, 0x4d, 0x72, 0x7b, 0x60, 0x69, 0x1e, 0x17, 0x0c, 0x05, 0x3a, 0x33, 0x28, 0x21,
                0x4f, 0x46, 0x5d, 0x54, 0x6b, 0x62, 0x79, 0x70, 0x07, 0x0e, 0x15, 0x1c, 0x23, 0x2a, 0x31, 0x38,
                0x41, 0x48, 0x53, 0x5a, 0x65, 0x6c, 0x77, 0x7e, 0x09, 0x00, 0x1b, 0x12, 0x2d, 0x24, 0x3f, 0x36,
                0x58, 0x51, 0x4a, 0x43, 0x7c, 0x75, 0x6e, 0x67, 0x10, 0x19, 0x02, 0x0b, 0x34, 0x3d, 0x26, 0x2f,
                0x73, 0x7a, 0x61, 0x68, 0x57, 0x5e, 0x45, 0x4c, 0x3b, 0x32, 0x29, 0x20, 0x1f, 0x16, 0x0d, 0x04,
                0x6a, 0x63, 0x78, 0x71, 0x4e, 0x47, 0x5c, 0x55, 0x22, 0x2b, 0x30, 0x39, 0x06, 0x0f, 0x14, 0x1d,
                0x25, 0x2c, 0x37, 0x3e, 0x01, 0x08, 0x13, 0x1a, 0x6d, 0x64, 0x7f, 0x76, 0x49, 0x40, 0x5b, 0x52,
                0x3c, 0x35, 0x2e, 0x27, 0x18, 0x11, 0x0a, 0x03, 0x74, 0x7d, 0x66, 0x6f, 0x50, 0x59, 0x42, 0x4b,
                0x17, 0x1e, 0x05, 0x0c, 0x33, 0x3a, 0x21, 0x28, 0x5f, 0x56, 0x4d, 0x44, 0x7b, 0x72, 0x69, 0x60,
                0x0e, 0x07, 0x1c, 0x15, 0x2a, 0x23, 0x38, 0x31, 0x46, 0x4f, 0x54, 0x5d, 0x62, 0x6b, 0x70, 0x79,
        };


        return crcTab7[(curCrc * 2) ^ byte];
}

static void vsdPrvGenerateCsd(struct VSD *vsd, uint8_t *dst)
{
	uint_fast8_t i = 0, j, crc = 0;

	if (vsd->isSDHC) {	//create a CSDv2

		dst[i++] = 0x40;							//CSD_STRUCTURE = 1
		dst[i++] = 0x0e;							//TAAC = as per spec
		dst[i++] = 0x00;							//NSAC = as per spec
		dst[i++] = 0x32;							//TRAN_SPEED = 25MHz
		dst[i++] = 0x5b;							//CCC.hi8 = as per spec
		dst[i++] = 0x59;							//CCC.lo4 = as per spec, READ_BL_LEN as per spec
		dst[i++] = 0x00;							//READ_BL_PARTIAL, WRITE_BLK_MISALIGN, READ_BLK_MISALIGN as per spec, DSR_IMP = 0
		dst[i++] = vsd->size.hc.cSize >> 16;		//C_SIZE.hi6 as per calc
		dst[i++] = vsd->size.hc.cSize >> 8;			//C_SIZE.mid8 as per calc
		dst[i++] = vsd->size.hc.cSize;				//C_SIZE.lo8 as per calc
		dst[i++] = 0x7f;							//ERASE_BLK_EN, SECTOR_SIZE.hi6 as per spec
		dst[i++] = 0x80;							//SECTOR_SIZE.lo1, WP_GRP_SIZE as per spec
		dst[i++] = 0x0a;							//WP_GRP_ENABLE, R2W_FACTOR, WRITE_BL_LEN.hi2 as per spec
		dst[i++] = 0x40;							//WRITE_BL_LEN.lo2, WRITE_BL_PARTIAL as per spec
	}
	else {				//create a CSDv1
		
		dst[i++] = 0x00;							//CSD_STRUCTURE = 0
		dst[i++] = 0x3b;							//TAAC = 3us
		dst[i++] = 0x10;							//NSAC = 16
		dst[i++] = 0x32;							//TRAN_SPEED = 25MHz
		dst[i++] = 0x5b;							//CCC.hi8 = as per spec
		dst[i++] = 0x50 + vsd->size.sc.readBlLen;	//CCC.lo4 = as per spec, READ_BL_LEN as per calc
		dst[i++] = vsd->size.sc.cSize >> 10;		//no partial or misaligned reads, C_SIZE.hi2 = as per calc
		dst[i++] = vsd->size.sc.cSize >> 2;			//C_SIZE.mid8 = as per calc
		dst[i++] = vsd->size.sc.cSize << 6;			//C_SIZE.lo2 = as per calc, currents all minimal
		dst[i++] = vsd->size.sc.cSizeMult >> 1;		//currents all minimal, C_SIZE_MULT.hi2 = as per calc
		dst[i++] = vsd->size.sc.cSizeMult << 7;		//C_SIZE_MULT.lo1 = as per calc, ERASE_BLK_EN = 0, SECTOR_SIZE.hi6 = 0
		dst[i++] = 0x00;							//SECTOR_SIZE.lo1 = 0, WP_GRP_SIZE = 0
		dst[i++] = 0x02;							//WP_GRP_ENABLE = 0, R2W_FACTOR = 0, WRITE_BL_LEN.hi2 = 2
		dst[i++] = 0x40;							//WRITE_BL_LEN.lo2 = 1, WRITE_BL_PARTIAL = 0
	}

	dst[i++] = 0x00;							//FILE_FORMAT_GRP = 0, COPY = 0, PERM_WRITE_PROTECT = 0, TMP_WRITE_PROTECT = 0, FILE_FORMAT = 0

	for (j = 0; j < i; j++)
		crc = vsdPrvCrcAccount(crc, dst[j]);

	dst[i++] = 2 * crc + 1;
}

static void vsdPrvGenerateCid(struct VSD *vsd, uint8_t *dst)
{
	uint_fast8_t i = 0, j, crc = 0;

	dst[i++] = 'D';		//MID = 'D'

	dst[i++] = 'g';		//OID = 'gr'
	dst[i++] = 'r';

	dst[i++] = 'v';		//PNM = 'vSDHC' or 'vSDSC'
	dst[i++] = 'S';
	dst[i++] = 'D';
	dst[i++] = vsd->isSDHC ? 'H' : 'S';
	dst[i++] = 'C';

	dst[i++] = 0x10;	//PRV = 1.0

	dst[i++] = 0x78;	//PSN = 0x12345678
	dst[i++] = 0x56;
	dst[i++] = 0x34;
	dst[i++] = 0x12;

	dst[i++] = 0x01;	//MDT = july 2024
	dst[i++] = 0x87;
	
	for (j = 0; j < i; j++)
		crc = vsdPrvCrcAccount(crc, dst[j]);

	dst[i++] = 2 * crc + 1;
}

static void vsdPrvSendR1(struct VSD *vsd, uint8_t r1Bits)
{
	uint8_t val = r1Bits | (vsd->isInited ? 0x00 : 0x01);

	vsd->cmd[0] = val;
	vsd->state = SdStateSendR1;
	vsd->nextState = SdStatePostCmdIdle;
}

static void vsdPrvQueueTxData(struct VSD *vsd, uint32_t len, uint32_t delay)
{
	if (delay) {

		vsd->nextState = SdStateSendWait;
		vsd->delayCounter = delay;
	}
	else {
		vsd->nextState = SdStateSendData;
	}
	vsd->numBytes = len;
}

static void vsdPrvProcessCmd(struct VSD *vsd)
{
	uint32_t param = (((uint32_t)vsd->cmd[1]) << 24) + (((uint32_t)vsd->cmd[2]) << 16) + (((uint32_t)vsd->cmd[3]) << 8) + vsd->cmd[4];
	uint_fast8_t i, crc = 0;
	bool crcValid, crcValidIfNeeded, wasAcmd = vsd->acmd;

	vsd->acmd = false;

	for (i = 0; i < sizeof(vsd->cmd) - 1; i++)
		crc = vsdPrvCrcAccount(crc, vsd->cmd[i]);
	crcValid = crc * 2 + 1 == vsd->cmd[i];
	crcValidIfNeeded = crcValid || (vsd->cmd[0] != 8 && !vsd->crcOn);	//cmd8 always needsd CRC on

	if (VERBOSE > 0)
		fprintf(stderr, "CMD %u(0x%08x), CRC %s\r\n", vsd->cmd[0] & 0x3f, param, crcValid ? "OK" : (crcValidIfNeeded ? "IRRELEVANT" : "BAD"));

	if (!crcValidIfNeeded)
		vsdPrvSendR1(vsd, R1_RSP_CRC_ERR);
	else if ((vsd->cmd[0] & 0xc0) != 0x40) {
		fprintf(stderr, "invalid command format 0x%02x with param 0x%08x\n", vsd->cmd[0], param);
		vsdPrvSendR1(vsd, R1_RSP_ILLEGAL_CMD);
	}
	else {
		switch (vsd->cmd[0] & 0x3f) {
			case 0:
				vsd->initCounter = INIT_COMMANDS_REQD;
				vsd->isInited = false;
				vsdPrvSendR1(vsd, 0);
				break;

			case 8:
				vsd->hadCmd8 = true;
				vsd->data[0] = 0;
				vsd->data[1] = 0;
				vsd->data[2] = vsd->cmd[3];
				vsd->data[3] = vsd->cmd[4];
				vsdPrvSendR1(vsd, 0);
				vsdPrvQueueTxData(vsd, 4, 0);
				break;
			
			case 9:		//get CSD
				if (!vsd->isInited) {
					vsdPrvSendR1(vsd, R1_RSP_ILLEGAL_CMD);
					break;
				}
				vsdPrvSendR1(vsd, 0);
				vsdPrvGenerateCsd(vsd, vsd->data + 1);
				vsd->data[0] = 0xfe;
				vsdPrvQueueTxData(vsd, 16 + 3, 10);
				break;
				
			case 10:	//get CID
				if (!vsd->isInited) {
					vsdPrvSendR1(vsd, R1_RSP_ILLEGAL_CMD);
					break;
				}
				vsdPrvSendR1(vsd, 0);
				vsdPrvGenerateCid(vsd, vsd->data + 1);
				vsd->data[0] = 0xfe;
				vsdPrvQueueTxData(vsd, 16 + 3, 20);
				break;
			
			case 17:	//read block
				if (!vsd->isInited) {
					if (VERBOSE > 0)
						fprintf(stderr, "rejecting read due to not inited\n");

					vsdPrvSendR1(vsd, R1_RSP_ILLEGAL_CMD);
					break;
				}
				if (!vsd->isSDHC) {
					if (param % 512) {
						if (VERBOSE > 0)
							fprintf(stderr, "rejecting read due to misalignment\n");
						vsdPrvSendR1(vsd, R1_RSP_ADDRESS_ERR);
						break;
					}
					param /= 512;
				}
				if (param >= vsd->numSec) {
					if (VERBOSE > 0)
						fprintf(stderr, "rejecting read due to OOB\n");
					vsdPrvSendR1(vsd, R1_RSP_ADDRESS_ERR);
					break;
				}
				if (!vsd->readF(vsd->userData, param, vsd->data + 1)) {
					
					if (VERBOSE > 0)
						fprintf(stderr, "rejecting read due to higher level error\n");

					vsdPrvSendR1(vsd, R1_RSP_ERZ_SEQ_ERR);		//as sane an answer as any
					break;
				}
				vsd->data[0] = 0xfe;
				vsdPrvSendR1(vsd, 0);
				vsdPrvQueueTxData(vsd, 512 + 3, 100);
				break;

			case 24:	//write block
				if (!vsd->isInited) {
					if (VERBOSE > 0)
						fprintf(stderr, "rejecting write due to not inited\n");

					vsdPrvSendR1(vsd, R1_RSP_ILLEGAL_CMD);
					break;
				}
				if (!vsd->isSDHC) {
					if (param % 512) {
						if (VERBOSE > 0)
							fprintf(stderr, "rejecting write due to misalignment\n");
						vsdPrvSendR1(vsd, R1_RSP_ADDRESS_ERR);
						break;
					}
					param /= 512;
				}
				if (param >= vsd->numSec) {
					if (VERBOSE > 0)
						fprintf(stderr, "rejecting write due to OOB\n");
					vsdPrvSendR1(vsd, R1_RSP_ADDRESS_ERR);
					break;
				}
				vsd->writeSecNo = param;
				vsdPrvSendR1(vsd, 0);
				vsd->nextState = SdStateDataRxWait;
				break;

			case 55:
				vsd->acmd = true;
				vsdPrvSendR1(vsd, 0);
				break;

			case 41:
				if (!wasAcmd)
					vsdPrvSendR1(vsd, R1_RSP_ILLEGAL_CMD);
				else {
					if (!vsd->isSDHC || (param & 0x40000000)) {	//HC card will only init with a host that claims to support that

						if (vsd->initCounter)
							vsd->initCounter--;
						else
							vsd->isInited = true;
					}
					vsdPrvSendR1(vsd, 0);
				}
				break;
			
			case 59:
				vsd->crcOn = !!(param & 1);
				vsdPrvSendR1(vsd, 0);
				break;

			default:
				fprintf(stderr, "unknown command 0x%02x with param 0x%08x\n", vsd->cmd[0], param);
				vsdPrvSendR1(vsd, R1_RSP_ILLEGAL_CMD);
				break;
		}
	}
}

static uint8_t vsdPrvSpiProvideByteF(void *userData)
{
	struct VSD *vsd = (struct VSD *)userData;
	uint8_t ret = 0xff;

	switch (vsd->state) {
		
		case SdStateSendData:
			//this is not fast and i do not care
			ret = vsd->data[0];
			if (VERBOSE > 2)
				fprintf(stderr, "send data ! 0x%02x (%u bytes left incl this one)\n", ret, vsd->numBytes);
			memmove(vsd->data + 0, vsd->data + 1, sizeof(vsd->data) - 1);
			if (!--vsd->numBytes)
				vsd->state = SdStatePostCmdIdle;
			break;

		case SdStateSendWait:
			if (VERBOSE > 2)
				fprintf(stderr, "send data _ 0x%02x\n", 0xff);
			if (vsd->delayCounter)
				vsd->delayCounter--;
			else
				vsd->state = SdStateSendData;
			break;

		case SdStateSendR1:
			ret = vsd->cmd[0];
			if (VERBOSE > 1)
				fprintf(stderr, " -> 0x%02x\r\n", ret);
			vsd->state = vsd->nextState;
			break;

		case SdStateBusy:
			if (vsd->numBytes) {
				vsd->numBytes--;
				ret = 0x00;
			}
			else {
				vsd->state = SdStatePostCmdIdle;
			}
			break;

		case SdStateDataRxedSendReply:
			ret = vsd->data[0];
			if (VERBOSE > 1)
				fprintf(stderr, " -> 0x%02x (data reply)\r\n", ret);

			if (ret == 0xe5) {	//if we had a success, we go busy, else no
				
				vsd->state = SdStateBusy;
				vsd->numBytes = BUSY_BYTES_POST_WRITE;
			}
			else
				vsd->state = SdStatePostCmdIdle;
			break;

		case SdStateCmdRx:
		case SdStatePostCmdIdle:
		case SdStateDeselected:
		case SdStateDataRx:
		case SdStateDataRxWait:
			break;
	}

	return ret;
}

static void vsdPrvSpiAcceptByteF(void *userData, uint8_t byte)
{
	struct VSD *vsd = (struct VSD *)userData;

	if (VERBOSE > 2)
		fprintf(stderr, "recv data   0x%02x\n", byte);

	switch (vsd->state) {

		case SdStateCmdRx:
			vsd->cmd[vsd->numBytes++] = byte;
			if (vsd->numBytes == sizeof(vsd->cmd)) {

				vsd->state = SdStatePostCmdIdle;
				vsdPrvProcessCmd(vsd);
			}
			break;

		case SdStatePostCmdIdle:
			break;

		case SdStateDataRxWait:
			if (byte == 0xff)
				break;
			else if (byte != 0xfe) {
				fprintf(stderr, "VSD: first data byte not the data marker of 0xFE. saw 0x%02x\n", byte);
				abort();
			}
			else {
				vsd->numBytes = 0;
				vsd->state = SdStateDataRx;
			}
			break;

		case SdStateDataRx:
			vsd->data[vsd->numBytes++] = byte;
			if (vsd->numBytes == 512 + 2) {
				vsd->data[0] = vsd->writeF(vsd->userData, vsd->writeSecNo, vsd->data) ? 0xe5 : 0xed;
				if (VERBOSE > 1)
					fprintf(stderr, " write to sec %u %s\r\n", vsd->writeSecNo, vsd->data[0] == 0xe5 ? "ACCEPTED" : "REJECTED");
				vsd->state = SdStateDataRxedSendReply;
			}
			break;

		case SdStateDeselected:
			fprintf(stderr, "VSD: data in deselected state\n");
			abort();
			break;

		case SdStateSendData:
		case SdStateSendWait:
		case SdStateDataRxedSendReply:
		case SdStateBusy:
		case SdStateSendR1:
			break;
	}
}

static void vsdPrvSpiSelectionChanged(void *userData, bool selected)
{
	struct VSD *vsd = (struct VSD *)userData;

	if (selected) {

		if (vsd->state != SdStateDeselected) {

			fprintf(stderr, "VSD: selected while not in deselected state\n");
			abort();
		}
		vsd->state = SdStateCmdRx;
		vsd->numBytes = 0;

		if (VERBOSE > 1)
			fprintf(stderr, "---------- SELECTED ----------\r\n");
	}
	else {

		switch (vsd->state) {

			case SdStateCmdRx:
				fprintf(stderr, "VSD: deselected while RXing command\n");
				abort();

			case SdStateDataRx:
				fprintf(stderr, "VSD: deselected while RXing data\n");
				abort();

			case SdStateDataRxedSendReply:					//due ot hos spi works we do not see this
				fprintf(stderr, "VSD: deselected before getting reply to uploaded data block\n");
				abort();

			case SdStateBusy:
				if (vsd->numBytes == BUSY_BYTES_POST_WRITE)	//due to how spi works, we see this
					fprintf(stderr, "VSD: deselected before getting reply to uploaded data block\n");
				else
					fprintf(stderr, "VSD: deselected while busy. technically ok, but a poor idea\n");
				abort();


			default:
				break;
		}

		if (VERBOSE > 1)
			fprintf(stderr, "----------DESELECTED---------- (state %u)\r\n", vsd->state);
		vsd->state = SdStateDeselected;
	}
}

struct VSD* vsdInit(VsdSecReadF readF, VsdSecWriteF writeF, void *userData, uint32_t numSec, bool forceHC)
{
	struct VSD *vsd;

	if (!numSec) {
		fprintf(stderr, "cannot represent a zero-sized card!\n");
		abort();
	}

	vsd = calloc(1, sizeof(struct VSD));
	if (vsd) {

		uint32_t effectiveNumSec;

		vsd->readF = readF;
		vsd->writeF = writeF;
		vsd->userData = userData;
		vsd->state = SdStateDeselected;
		vsd->crcOn = true;
		vsd->initCounter = INIT_COMMANDS_REQD;

		//sort out card size and calculate size register values for ease, round as needed

		vsd->isSDHC = forceHC || numSec > 4153344;	//as per spec this is the max size for an SDSC
		if (vsd->isSDHC) {				//SDHC sector count must be a multiple of 1024

			uint32_t numLargeBlocks = numSec / 1024;

			if (!numLargeBlocks) {
				fprintf(stderr, "card this small (%u secs) cannot be represented by an SDHC CSD register\n", numSec);
				abort();
			}

			vsd->size.hc.cSize = numLargeBlocks - 1;
			effectiveNumSec = 1024 * (vsd->size.hc.cSize + 1);
			fprintf(stderr, "SDHD card sizing: %u sectors, C_SIZE=%u\n", effectiveNumSec, vsd->size.hc.cSize);
		}
		else {							//SD sector count is complex

			uint32_t numZeroes;

			//make C_SIZE fit
			for(numZeroes = 2; (numSec >> numZeroes) > 0x1000; numZeroes++);
			vsd->size.sc.cSize = (numSec >> numZeroes) - 1;

			//allocate zeroes: first to C_SIZE_MULT
			numZeroes -= 2;		//C_SIZE_MULT if 0 means "2"
			vsd->size.sc.cSizeMult = (numZeroes > 7) ? 7 : numZeroes;
			numZeroes -= vsd->size.sc.cSizeMult;

			//allocate zeroes: now to READ_BL_LEN (max one)
			if (numZeroes > 1) {
				fprintf(stderr, "card this large (%u secs) cannot be represented by an SDSC CSD register\n", numSec);
				abort();
			}
			vsd->size.sc.readBlLen = 9 + numZeroes;
			effectiveNumSec = (vsd->size.sc.cSize + 1) << (vsd->size.sc.cSizeMult + vsd->size.sc.readBlLen + 2 - 9);
			fprintf(stderr, "SDSC card sizing: %u sectors, C_SIZE=%u, C_SIZE_MULT=%u, READ_BL_LEN=%u\n", effectiveNumSec, vsd->size.sc.cSize, vsd->size.sc.cSizeMult, vsd->size.sc.readBlLen);
		}

		if (effectiveNumSec != numSec)
			fprintf(stderr, "VSD: card truncated from %u sec to %u sec to allow for proper CSD representation\n", numSec, effectiveNumSec);

		vsd->numSec = effectiveNumSec;
		vsd->vspi = vspiInit("VSD", SpiMode0);
		
		if (vsd->vspi) {

			vspiDeviceRegister(vsd->vspi, vsdPrvSpiProvideByteF, vsdPrvSpiAcceptByteF, vsdPrvSpiSelectionChanged, vsd);
			return vsd;
		}
		free(vsd);
		vsd = NULL;
	}
	return vsd;
}

struct VSPI* vsdGetVSPI(struct VSD* vsd)
{
	return vsd->vspi;
}

void vsdDeinit(struct VSD* vsd)
{
	vspiDestroy(vsd->vspi);
	free(vsd);
}



