#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/select.h>
#include <signal.h>
#include <curses.h>
#include "soc.h"
#include "cpu.h"


static FILE *mDiskF;
static uint32_t mDiskSecs;
static uint8_t mRom[8192], mRam[0x300], mRamSta[sizeof(mRam) / 4];		//only the lower nibble of ram bytes is used!


extern uint32_t numMipsCycles;
extern uint64_t totalCy;

void socExtPutChar(char ch)
{
	while (write(1, &ch, 1) < 0 && errno == EINTR);
}

int_fast16_t socExtGetChar(void)
{
	return -1;
}

bool socExtSdSecRead(uint32_t sec, uint8_t *dst)
{
	fseek(mDiskF, 512 * sec, SEEK_SET);
	return fread(dst, 1, 512, mDiskF) == 512;
}

bool socExtSdSecWrite(uint32_t sec, const uint8_t *dst)
{
	fseek(mDiskF, 512 * sec, SEEK_SET);
	return fwrite(dst, 1, 512, mDiskF) == 512;
}

uint32_t socExtSdGetSize(void)
{
	return mDiskSecs;
}

static void usage(const char *self)
{
	fprintf(stderr, "USAGE: %s <ROMIMG> <DISKIMG>\n", self);
	exit(-1);
}

void ctl_cHandler(int v)	//handle SIGTERM      
{
	
	(void)v;
	int i;

	endwin();

	fclose(mDiskF);
	fprintf(stderr, "\nExiting after %u MIPS cycles using %llu 4004 instr cycles (%llu instrs) = %u per (%llu clock cycles, %llu seconds of realtime, which is %f days). Thus a 750 KHz 4004 emulates a %0.3f Hz R3000\n",
		(unsigned)numMipsCycles, (unsigned long long)totalCy, (unsigned long long)cpuGetInstrCt(), (unsigned)(totalCy / numMipsCycles), (unsigned long long)totalCy * 8, (unsigned long long)totalCy * 8 / 750000, (double)totalCy * 8.0 / 750000 / (3600*24), 750000.0 / 8 * numMipsCycles / totalCy);
	
	for (i = -1; i < 2; i++) {

		uint64_t psramNumSelectionsP0 = socPrvGetPsramNumSelectionsPage0(i);
		uint64_t psramNumSelections = socPrvGetPsramNumSelections(i);
		uint64_t psramSelectedTicks = socPrvGetPsramSelectedTicks(i);
		uint32_t psramLongestSelectedTicks = socPrvGetPsramLongestSelectedTicks(i);
		const char *names[] = {"all", "0", "1"};

		fprintf(stderr, "PSRAM.%s was selected %llu times (%llu times by ROM page0) for an average of %u instr cycles (%f msec at 750KHz), max %u cycles (%f msec)\n", names[i + 1],
			(unsigned long long)psramNumSelections, (unsigned long long)psramNumSelectionsP0, (unsigned)((psramSelectedTicks + psramNumSelections / 2) / psramNumSelections), (double)psramSelectedTicks * 8.0 / (double)psramNumSelections / 750.0,
			psramLongestSelectedTicks, (double)psramLongestSelectedTicks * 8.0 / 750.0);
	}

	//profiling data
	static uint16_t addrs[8192];
	static uint64_t counts[8192];
	int j, k;

	//get
	memcpy(counts, cpuPrvGetProfilingData(), sizeof(counts));
	for (i = 0; i < sizeof(addrs) / sizeof(*addrs); i++)
		addrs[i] = i;

	//sort
	for (i = 0; i < 8192; i++) {
		uint64_t tc;
		uint16_t ta;

		k = i;
		for (j = i + 1; j < 8192; j++) {
			if (counts[j] > counts[k])
				k = j;
		}
		tc = counts[i];
		ta = addrs[i];
		counts[i] = counts[k];
		addrs[i] = addrs[k];
		counts[k] = tc;
		addrs[k] = ta;
	}

	//show top few hundred. code assuems that our sort was stable and equal counts will be sorted by increasing addr (which they will)
	fprintf(stderr, "TOP hot instrs (dupes skipped:\n");
	uint16_t prevAddr = 0xffff;
	uint64_t prevCount = 0xffffffff;
	unsigned rank = 0;
	for (i = 0; i < 8192 && rank < 100; i++) {

		if ((prevAddr + 1 != addrs[i] && prevAddr + 2 != addrs[i]) || prevCount != counts[i])
			fprintf(stderr, "  rank %3u: %u.0x%03x, ran %llu times\n", ++rank, addrs[i] >> 12, addrs[i] & 0xfff, (unsigned long long)counts[i]);

		prevAddr = addrs[i];
		prevCount = counts[i];
	}

	_exit(0);
}

int main(int argc, char **argv)
{
	unsigned long romSz, diskSz;
	FILE *romF;

	if (argc != 3 || !(romF = fopen(argv[1], "rb")) || !(mDiskF = fopen(argv[2], "r+b")))
		usage(argv[0]);

	fseek(romF, 0, SEEK_END);
	romSz = ftell(romF);
	rewind(romF);

	if (romSz > sizeof(mRom)) {
		fprintf(stderr, "ROM size of %lu if longer than allowed (%u)\n", romSz, (unsigned)sizeof(mRom));
		return -1;
	}

	if (romSz != fread(mRom, 1, sizeof(mRom), romF)) {
		fprintf(stderr, "ROM read failed\n");
		return -1;
	}
	fclose(romF);

	fseek(mDiskF, 0, SEEK_END);
	diskSz = ftell(mDiskF);
	
	if (diskSz % DISK_SEC_SZ) {
		fprintf(stderr, "DISK size of %lu is not a multiple of sector size (%u)\n", romSz, (unsigned)DISK_SEC_SZ);
		return -1;
	}

	mDiskSecs = diskSz / DISK_SEC_SZ;

	//setup the terminal
	initscr();
	cbreak();
	noecho();
	clear();

	if (!socInit(mRom, romSz, mRam, mRamSta, sizeof(mRam))) {
		fprintf(stderr, "Failed to init SoC\n");
		endwin();
		return -1;
	}

	signal(SIGINT, &ctl_cHandler);

	socRun();

	return 0;
}