/* Borg Solitaire for LCARS 24, based on Klondike Solitaire
   * Face cards have pictures of Voyager crew members, including the
     Doctor, plus Q and Barklay as Borg!
   * Names of some of the arrays and functions are based on
     the traditional nomenclature for the various piles of cards
     in Klondike Solitaire:
      Stock: dealer's face-down pile
      Waste: face-up pile of cards dealt
      Foundations: four piles at top filled by suit
      Tableaus: seven piles below filled by alternating colors
            (abbreviated to Tab in the code)
   * First public release 07/26/03
*/
/* This program is free software. You may redistribute and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License or
 * (at your option) any later version.
 *
 * Refer to the file C:\LCARS24\DATA\COPYING.TXT for details.
 */


#include "borg.h"

#define OK 1
#define NG 0
/* values for UndoMode, other than NG */
#define DEAL 1
#define WASTE_TO_TAB 2
#define WASTE_TO_FOUNDATION 3
#define TAB_TO_FOUNDATION 4
#define FOUNDATION_TO_TAB 5
#define TAB_TO_TAB 6

/* Standard Bicycle cards are 63 x 89 mm, so multiplying by 1.1,
   we get proportional numbers of pixels for cw (card width)
   and ch card height below, where 69 is the maximum
   convenient width for the program. The dark outline
   adds 2, but the real dimensions are 69 x 98. Windows
   uses 69 x 94, which makes them look a bit squatty.
   Face-down, the cards are drawn narrower because of the
   black background, since the dark outline of the face-up
   cards is only seen where it overlaps another card.
*/

int OldSolW1x;
int OldSolW1y;
int SolW1x;
int SolW1y;
int SolDeck[52];
int TabDown[7][6];
int TabDownCount[7];
int OldTabDownCount[7];
int TabUp[7][26];
int TabUpCount[7];
int SolStock[24];
int SolStockCount;
int SolStockUndealt;
int Waste[24];
int WasteCount;
int WasteShowing;
int PrevSolStock[24];
int PrevSolStockCount;
int PrevSolStockUndealt;
int PrevWaste[24];
int PrevWasteCount;
int PrevWasteShowing;
int Foundation[4][14];
int FoundationCount[4];
int SolCardString[13];
int SolUndoMode;
int UndoTabNum;
int UndoSourceTabNum;
int UndoLength;
int UndoFoundationNum;
int SolGameOver;
int RedoSignage;


void SolBox1a(int Hue)
{
	int boxtop;

	boxtop=10;
	rectfill(screen,10,boxtop,110,boxtop+54,Hue);
	PutsLuna(14,boxtop + 37, BLACK,Hue, "CLOSE");
	PutsLuna(91,boxtop + 37, BLACK,Hue, "F12");
}
void SolBox2a(int Hue)
{
	int boxtop;

	boxtop=68;
	rectfill(screen,10,boxtop,110,boxtop+54,Hue);
}
void SolBox1b(int Hue)
{
	int boxtop;

	boxtop=590 - 54;
	rectfill(screen,10,boxtop,110,boxtop+53,Hue);
	PutsLuna(14,boxtop + 4, BLACK,Hue, "NEW GAME");
	PutsLuna(96,boxtop + 4, BLACK,Hue, "F1");
}
void SolBox2b(int Hue)
{
	int boxtop;

	boxtop=590 - (54*2) - 3;
	rectfill(screen,10,boxtop,110,boxtop+53,Hue);
	PutsLuna(14,boxtop + 4, BLACK,Hue, "MOVE");
	PutsLuna(14 + 29,boxtop + 37, BLACK,Hue, "ARROW KEYS");
}
void SolBox3b(int Hue)
{
	int boxtop;

	boxtop=590 - (54*3) - 6;
	rectfill(screen,10,boxtop,110,boxtop+53,Hue);
	PutsLuna(14,boxtop + 4, BLACK,Hue, "PICK UP/DROP");
	PutsLuna(96-16,boxtop + 37, BLACK,Hue, "ENTER");
}
void SolBox4b(int Hue)
{
	int boxtop;

	boxtop=590 - (54*4) - 9;
	rectfill(screen,10,boxtop,110,boxtop+53,Hue);
	PutsLuna(14,boxtop + 4, BLACK,Hue, "TO TOP");
	PutsLuna(83,boxtop + 4, BLACK,Hue, "PGDN");
}
void SolBox5b(int Hue)
{
	int boxtop;

	boxtop=590 - (54*5) - 12;
	rectfill(screen,10,boxtop,110,boxtop+53,Hue);
	PutsLuna(14,boxtop + 4, BLACK,Hue, "ALL TO TOP");
	PutsLuna(84,boxtop + 4, BLACK,Hue, "PGUP");
}
void SolElbowSignage(int Hue1, int Hue2)
{
	int boxtop;

	boxtop=148;
	PutsLuna(14,boxtop + 6 -24, BLACK,Hue1, "DEAL");
	PutsLuna(80,boxtop + 6 -24, BLACK,Hue1, "ENTER");

	boxtop=590 - (54*6) - 16;
	PutsLuna(14,boxtop + 28 +9, BLACK,Hue2, "HELP");
	PutsLuna(91,boxtop + 28 +9, BLACK,Hue2, "F11");
}
void SolBoxes()
{
	SolBox1a(LILAC);
	SolBox2a(PALEBLUE);
	SolBox1b(MANGO);
	SolBox2b(MANGO);
	SolBox3b(MANGO);
	SolBox4b(BRICK);
	SolBox5b(BRICK);
	SolElbowSignage(MANGO, PALEBLUE);
}
int Sol24Help()
{
	int Result;

	strcpy(CurrentFile, "C:\\LCARS24\\LC24APPS\\SOL24\\SOL24.TXT");
	Result=OpenHelp(CurrentFile);
	min_old=70;
	return(Result);
}
void SolW1Cursor()
{
	int gx,gy,gx2;

	gx=62+cw+(OldSolW1x*90);
	gx2=gx;
	if(OldSolW1y==1) gy=120;
	if(OldSolW1y==1 && OldSolW1x==2) gx += 28;
	if(OldSolW1y > 1) gy=260+(TabDownCount[SolW1x-1]*3)+(OldSolW1y*15)-30;
	EnterSymbol(gx,gy,1,BLACK);
/* Instead of a complex formula, just erase along the whole tableau. */
	rectfill(screen,gx2,258,gx2+16,530,BLACK);
	gx=62+cw+(SolW1x*90);
	if(SolW1y==1) gy=120;
	if(SolW1y==1 && SolW1x==2) gx += 28;
	if(SolW1y > 1) gy=260+(TabDownCount[SolW1x-1]*3)+(SolW1y*15)-30;
	EnterSymbol(gx,gy,1,NAPLES);
	OldSolW1x=SolW1x;
	OldSolW1y=SolW1y;
}
int DrawSpot(int x, int y)
{
	int a,b,i, NAPLES2, NAPLES4;

	NAPLES2=makecol(225,180,135);
	NAPLES4=makecol(145,116,87);

	if(x > 200) {				/* a foundation */
		for(b=0; b<25; ++b) {
			for(a=0; a<18; ++a) {
				putpixel(screen,x+1+(a*4)+2,y+1+(b*4),NAPLES4);
			}
		}

		i=0;
		for(b=0; b<25; ++b) {
			for(a=0; a<9; ++a) {
				putpixel(screen,x+1+(a*8)+i,y+1+(b*4)+2,NAPLES4);
			}
		if(i==0) i=4;
			else i=0;
		}
		vline(screen,x+cw,y,y+ch,BLACK);
	}
	--cw;
	hline(screen,x+2,y,x+cw-2,NAPLES2);
	hline(screen,x+2,y+ch,x+cw-2,NAPLES2);
	vline(screen,x,y+2,y+ch-2,NAPLES2);
	vline(screen,x+cw,y+2,y+ch-2,NAPLES2);
	putpixel(screen,x+1,y+1,NAPLES2);
	putpixel(screen,x+cw-1,y+1,NAPLES2);
	putpixel(screen,x+1,y+ch-1,NAPLES2);
	putpixel(screen,x+cw-1,y+ch-1,NAPLES2);
	++cw;
}
void SolUndoOn()
{
	int boxtop;

	boxtop=68;
	PutsLuna(14,boxtop + 37, BLACK,PALEBLUE, "UNDO");
	PutsLuna(96,boxtop + 37, BLACK,PALEBLUE, "BS");
}
void SolUndoOff()
{
	int boxtop;

	boxtop=68;
	rectfill(screen,10,boxtop,110,boxtop+54,PALEBLUE);
}
void EraseTab(int TabNum)
{
	rectfill(screen,150+(TabNum*90),256,150+(TabNum*90)+cw+17,590,BLACK);
}
int DrawPregnantTab(int TabNum, int NumCards)
{
	int a;

	if(NumCards==0) return(0);
	for(a=0; a<NumCards; ++a) DrawDownCard(150+(90*TabNum),264+(a*3),0);
	DrawDownCard(150+(90*TabNum),264+(a*3),1); /* redo top one with design */
}
int DrawTab(int TabNum)
{
	int NumDown, NumUp, a;

	NumDown=TabDownCount[TabNum];
	NumUp=TabUpCount[TabNum];
	rectfill(screen,150+(TabNum*90)+cw+1,256,150+(TabNum*90)+cw+17,590,BLACK);
	if(NumDown + NumUp==0) {
		EraseTab(TabNum);
		return(1);
	}
	if(NumDown > 0 && NumUp==0) {
		EraseTab(TabNum);
		for(a=0; a<NumDown-1; ++a) DrawDownCard(150+(90*TabNum),264+(a*3),0);
			DrawDownCard(150+(90*TabNum),264+(a*3),1); /* redo top one */
		return(2);
	}
	EraseTab(TabNum);
	for(a=0; a<NumDown; ++a) DrawDownCard(150+(90*TabNum),264+(a*3),0);
	for(a=0; a<NumUp; ++a) {
		DrawCard(150+(90*TabNum),264+(NumDown*3)+(a*15),
				TabUp[TabNum][a]);
	}
}
int DrawSplitTab(int TabNum)
{
	int NumDown, NumUp, a;

	NumDown=TabDownCount[TabNum];
	NumUp=TabUpCount[TabNum];
	rectfill(screen,150+(TabNum*90)+cw+1,256,150+(TabNum*90)+cw+17,590,BLACK);
	if(NumDown + NumUp==0) {
		EraseTab(TabNum);
		return(1);
	}
	if(NumDown > 0 && NumUp==0) {
	for(a=0; a<NumDown-1; ++a) DrawDownCard(150+(90*TabNum),264+(a*3),0);
	DrawDownCard(150+(90*TabNum),264+(a*3),1); /* redo top one */
		return(2);
	}
	EraseTab(TabNum);
	for(a=0; a<NumDown-1; ++a) DrawDownCard(150+(90*TabNum),264+(a*3),0);
	if(NumDown > 0) DrawDownCard(150+(90*TabNum),264+(a*3),1);
	for(a=0; a<SolW1y-2; ++a) {
		DrawCard(150+(90*TabNum),264+(NumDown*3)+(a*15),TabUp[TabNum][a]);
	}
}
int TransferOK(int Dest, int Guest)
{	 /* decide whether a tableau can accept a card */
	/* return 0 if OK, 1 if not */
	int Host=TabUp[Dest][TabUpCount[Dest]-1];

				/* clear pavement: kings only */
	if(TabDownCount[Dest] + TabUpCount[Dest]==0) {
		if(Guest==13 || Guest==26 || Guest==39 || Guest==52) return(0);
		else return(1);
	}
				/* not blank to make it here, but nothing face-up */
	if(TabUpCount[Dest]==0) return(1);
				/* Don't park on an ace, Dummy! */
	if(Host==1 || Host==14 || Host==27 || Host==40) return(1);
				/* the main thing */
	if(Guest - Host==12) return(0);
	if(Guest - Host==38) return(0);
	if(Host - Guest==14) return(0);
	if(Host - Guest==40) return(0);
	return(1);
}
int DrawCardString(int x, int y)
{
	int b;
	int Length=0;

	for(b=SolW1y; b <= TabUpCount[SolW1x-1]+1; ++b) {
		DrawCard(x,y+((b-2)*15)-((SolW1y-2)*15), TabUp[SolW1x-1][b-2]);
		SolCardString[Length]=TabUp[SolW1x-1][b-2];
		++Length;
	}
	return(Length);
}
int DragFromTab()
{
	BITMAP *bmp=create_bitmap(710,350);
	int PlayerCard,Result,z,TabNum,dx,dy,sx,sy,Dest,a,b;
	int Length,Holder,p1,p2,p3;


	Holder=SolW1y;
	TabNum=SolW1x-1;
	Dest=SolW1x-1;
	PlayerCard=TabUp[TabNum][TabUpCount[TabNum]-1];
	EraseTab(TabNum);
	--TabUpCount[TabNum];
	DrawSplitTab(TabNum);
	++TabUpCount[TabNum];
	if(SolW1y > TabUpCount[SolW1x-1] + 1) SolW1y=TabUpCount[SolW1x-1] + 1;
	dx=160+(90*TabNum);

	dy=279+(TabDownCount[TabNum]*3)+((SolW1y-2)*15);
	clear_bitmap(bmp);
	blit(screen, bmp, 140, 230, 0, 0, 710, 580);

/* This just speeds up the pickup, not yet needing background. */
	sx=90;
	sy=18;
	goto express;

loopback:
	vsync();
	blit(bmp, screen, 0, 0, 140, 230, 710, 580);
express:
	if(SolW1y > TabUpCount[SolW1x-1] + 1) SolW1y=TabUpCount[SolW1x-1] + 1;
	Length=DrawCardString(dx,dy);
	EnterSymbol(dx+cw-10,dy-4,1,ORANGE);
loopback2:

	do {
		SolPutTime();
	} while(!kbhit());
	c=GetKey();
	switch(c)
		{
			case 13:
				vsync();
				if(TransferOK(Dest,SolCardString[0])==0) {
					p1=TabUpCount[Dest];
					p2=p1+Length;
					p3=0;
					for(b=p1; b < p2; ++b) {
						TabUp[Dest][b]=SolCardString[p3];
						++p3;
					}
					TabUpCount[TabNum] -= Length;
					TabUpCount[Dest]	+= Length;
					UndoTabNum=Dest;
					UndoSourceTabNum=TabNum;
					UndoLength=Length;
					DrawTab(Dest);
					DrawTab(TabNum);
					SolW1x=Dest+1;
					UndoTabNum=Dest;
					UndoSourceTabNum=TabNum;
					UndoLength=Length;
					SolUndoMode=TAB_TO_TAB;
					SolUndoOn();
				} else {
					vsync();
					blit(bmp, screen, 0, 0, 140, 230, 710, 580);
					DrawTab(TabNum);
				}
				destroy_bitmap(bmp);
				SolW1y=2;
				SolW1Cursor();
				return(Result);
			case F12: return(86); /* three steps to main to exit */
			case F1:	 goto loopback2;
			case F9:	 /* SWReport(); */
					goto loopback;
			case F8:
					goto loopback;
			case UPARROW:
				if(Length > 1) goto loopback2;
				if(dy>278) {
					TabToFoundation(TabNum);
					vsync();
					blit(bmp, screen, 0, 0, 140, 230, 710, 580);
					destroy_bitmap(bmp);
					DrawTab(TabNum);
					--SolW1y;
					if(SolW1y<2) SolW1y=2;
					SolW1Cursor();
					return(Result);
				} else goto loopback2;
/*
				if(SolW1y > 2) {
					dy-=sy;
					++SolW1y;
				}
*/
				goto loopback2;
			case DOWNARROW:
				goto loopback2;
			case LEFTARROW:
				if(dx < 170) goto loopback2;
				a=dx-sx;
				b=36/Length; /* less animation with longer loads */
				for(z=1; z < b; ++z) {
					dx-=(z*.1);
					DrawPhantom(dx,dy,Length);
				}
				dx=a;
				--Dest;
				goto loopback;
			case RIGHTARROW:
				if(dx > 630) goto loopback2;
				b=dx;
				a=dx+sx;
				b=36/Length; /* less animation with longer loads */
				for(z=1; z < b; ++z) {
					dx+=(z*.1);
					DrawPhantom(dx,dy,Length);
				}
				dx=a;
				++Dest;
				goto loopback;
			case 350: Screenshot792("Borg8.bmp");
					PanelBeep();
					goto loopback;
			case 351: Screenshot1024("Borg10.bmp");
					PanelBeep();
					goto loopback;
			default: /* ScoldUser(); */
					goto loopback;
		}
cursor:
	SolW1Cursor();
	goto loopback;
}
int DragFromWaste()
{	 /* Come here only if SolW1y==1 */
	BITMAP *bmp=create_bitmap(710,500);
	int PlayerCard,Result,z,TabNum,dx,dy,sx,sy,Dest,a,b;
	int First=1;
	int JumpRight=0;


		if(WasteCount==0) return(1);
		PlayerCard=Waste[WasteCount-1];
		--WasteCount;
		--WasteShowing;
		if(WasteCount < 3) WasteShowing=WasteCount;
		rectfill(screen,240,80,300+cw,95+ch,BLACK); /* erase */
		UpdateWasteDisplay();
		clear_bitmap(bmp);
		blit(screen, bmp, 140, 80, 0, 0, 300, 340);
		dx=290;
		dy=94;
		DrawCard(dx,dy,PlayerCard);
		EnterSymbol(dx+cw-10,dy-4,1,ORANGE);
loop1:
	do {
		SolPutTime();
	} while(!kbhit());
	c=GetKey();
	switch(c)
		{
			case 13: /* cancel and return */
				rectfill(screen,240,80,300+cw,95+ch,BLACK); /* erase */
				++WasteCount;
				++WasteShowing;
				UpdateWasteDisplay();
				SolW1Cursor();
				return(1);
			case DOWNARROW: /* bring the card down */
alternate:		a=ch;
				b=dy;
				for(z=1; z < 12; ++z) { /* max: 16 */
					dy+=z;
					DrawCard(dx,dy,0);
				}
				ch=a;
				blit(bmp, screen, 0, 0, 140, 80, 300, 340);
				goto bringdown;
			case LEFTARROW: /* go away and deal more cards */
				rectfill(screen,240,80,300+cw,95+ch,BLACK); /* erase */
				++WasteCount;
				++WasteShowing;
				UpdateWasteDisplay();
				SolW1x=1;
				SolW1Cursor();
				return(1);
			case RIGHTARROW: /* try sending the card to a foundation */
				++WasteCount;
				++WasteShowing;
				Result=WasteToFoundation();
				if(Result==0) {
					rectfill(screen,240,80,300+cw,95+ch,BLACK);
					UpdateWasteDisplay();
					SolW1Cursor();
					return(1);
				} else
					--WasteCount;
					--WasteShowing;
					JumpRight=1; /* Slide to right after going down */
					goto alternate; /* send the card down */
			default:  /* ScoldUser(); */
					goto loop1;
		}

bringdown:
	TabNum=2;
	Dest=2;
	SolW1y=2;

	dy=234+(TabDownCount[TabNum]*3)+(TabUpCount[TabNum]);
	clear_bitmap(bmp);
	blit(screen, bmp, 140, 230, 0, 0, 710, 500);

/* This just speeds up the pickup, not yet needing background refresh. */
	sx=90;
	sy=18;
	goto express;

loopback:
	vsync();
	blit(bmp, screen, 0, 0, 140, 230, 710, 580);
express:
	DrawCard(dx,dy,PlayerCard);
	EnterSymbol(dx+cw-10,dy-4,1,ORANGE);
	if(JumpRight==1) goto slide_right;
		else {
			if(First==1) {
				Dest=1;
				dx=160+(90*Dest);
				First=0;
				goto loopback;
			}
		}
loopback2:
	do {
		SolPutTime();
	} while(!kbhit());
	c=GetKey();
	switch(c)
		{
			case 13:
				vsync();
				blit(bmp, screen, 0, 0, 140, 230, 710, 580);
				if(TransferOK(Dest,PlayerCard)==0) {
					TabUp[Dest][TabUpCount[Dest]]=PlayerCard;
					++TabUpCount[Dest];
					DrawTab(Dest);
					SolW1x=Dest+1;
					SolW1y=2;
					UndoTabNum=Dest;
					SolUndoMode=WASTE_TO_TAB;
					SolUndoOn();
				} else {
					vsync();
					blit(bmp, screen, 0, 0, 140, 230, 710, 580);
					++WasteCount;
					++WasteShowing;
					UpdateWasteDisplay();
					SolW1y=1;
					SolW1x=2;
				}
				destroy_bitmap(bmp);
				SolW1Cursor();
				return(Result);
			case F12: return(86); /* three step to main to exit */
			case F1: SolNewGame();
					goto loopback;
			case F9: /*	 SWReport(); */
					goto loopback;
			case UPARROW:
				if(SolW1y ==2) {
					rectfill(screen,240,80,300+cw,95+ch,BLACK);
					++WasteCount;
					++WasteShowing;
					UpdateWasteDisplay();
					SolW1x=2;
					SolW1y=1;
					SolW1Cursor();
					vsync();
					blit(bmp, screen, 0, 0, 140, 230, 710, 580);
					destroy_bitmap(bmp);
					return(1);
				}
				--SolW1y;
				dy-=sy;
				goto loopback;
			case DOWNARROW:
				if(dy > 500) goto loopback2;
				++SolW1y;
				dy+=sy;
				goto loopback;
			case LEFTARROW:
				if(dx < 170) goto loopback2;
slide_left:		 if(First==1) {
					First=0;
					--Dest;
					dx=160+(90*Dest);
					goto loopback;
				}
				dx-=sx;
				--Dest;
				goto loopback;
			case RIGHTARROW:
				if(dx > 630) goto loopback2;
slide_right:			if(First==1) {
					First=0;
					dx=160+(90*Dest);
					goto loopback;
				}
				dx+=sx;
				++Dest;
				JumpRight=0;
				goto loopback;
			case 350: Screenshot792("Borg8.bmp");
					PanelBeep();
					goto loopback;
			case 351: Screenshot1024("Borg10.bmp");
					PanelBeep();
					goto loopback;
			default: /* ScoldUser(); */
					goto loopback;
		}
cursor:
	SolW1Cursor();
	goto loopback;
}
int DragFromFoundation()
{	 /* Come here only if SolW1y==1 */
	BITMAP *bmp=create_bitmap(800,600);
	int PlayerCard,Result,z,TabNum,dx,dy,sx,sy,Dest,FoundNum,OrigX,a,b;

	FoundNum=SolW1x-4;
	OrigX=SolW1x;
	if(FoundationCount[FoundNum]==0) return(1);

	PlayerCard=Foundation[FoundNum][FoundationCount[FoundNum]];

	if(FoundationCount[FoundNum]==1) {
	rectfill(screen,150+(90*(OrigX-1)),84,150+(90*(OrigX))-1,95+ch,BLACK);
		DrawSpot(150+(90*(OrigX-1)),84);
	} else {
	rectfill(screen,120+(90*(OrigX)),84,150+(90*(OrigX))-1,95+ch,BLACK);
		DrawCard(150+(90*(OrigX-1)),84,PlayerCard-1);
	}
	clear_bitmap(bmp);
	blit(screen, bmp, 0, 0, 0, 0, 800, 600);
	dx=160+(90*(OrigX-1));
	dy=94;
	DrawCard(dx,dy,PlayerCard);
	EnterSymbol(dx+cw-10,dy-4,1,ORANGE);

loop1:
	do {
		SolPutTime();
	} while(!kbhit());
	c=GetKey();
	switch(c)
		{
			case 13: /* cancel and return */
				blit(screen, bmp, 0, 0, 0, 0, 800, 600);
				destroy_bitmap(bmp);
	rectfill(screen,150+(90*(OrigX-1)),84,150+(90*(OrigX))-1,95+ch,BLACK);
				DrawCard(150+(90*(OrigX-1)),84,PlayerCard);
				SolW1Cursor();
				return(1);
			case DOWNARROW: /* bring the card down */
				a=ch; /* Lets RIGHTARROW default to here */
				b=dy;
				for(z=1; z < 12; ++z) { /* max: 16 */
					dy+=z;
					DrawCard(dx,dy,0);
				}
				ch=a;
				blit(bmp, screen, 0, 0, 0, 0, 800, 600);
				goto bringdown;
			default: /* ScoldUser(); */
				goto loop1;
		}

bringdown:
	TabNum=OrigX;
	Dest=OrigX-1;
	SolW1y=2;

	dy=234+(TabDownCount[TabNum]*3)+(TabUpCount[TabNum]);
	clear_bitmap(bmp);
	blit(screen, bmp, 0, 0, 0, 0, 800, 600);

/* This just speeds up the pickup, not yet needing background refresh. */
	sx=90;
	sy=18;
	goto express;

loopback:
	vsync();
	blit(bmp, screen, 0, 0, 0, 0, 800, 600);
express:
	DrawCard(dx,dy,PlayerCard);
	EnterSymbol(dx+cw-10,dy-4,1,ORANGE);
loopback2:
	do {
		SolPutTime();
	} while(!kbhit());
	c=GetKey();
	switch(c)
		{
			case 13:
				vsync();
				blit(bmp, screen, 0, 0, 0, 0, 800, 600);
				if(TransferOK(Dest,PlayerCard)==0) {
					TabUp[Dest][TabUpCount[Dest]]=PlayerCard;
					++TabUpCount[Dest];
					--FoundationCount[FoundNum];
					DrawTab(Dest);
					UndoFoundationNum=FoundNum;
					UndoTabNum=Dest;
					SolUndoMode=FOUNDATION_TO_TAB;
					SolUndoOn();
					SolW1x=Dest+1;
					SolW1y=2;
				} else {
					vsync();
					blit(bmp, screen, 0, 0, 0, 0, 800, 600);
					SolW1y=1;
					SolW1x=OrigX;
	rectfill(screen,150+(90*(OrigX-1)),84,150+(90*(OrigX))-1,95+ch,BLACK);
					DrawCard(150+(90*(OrigX-1)),84,PlayerCard);
				}
				destroy_bitmap(bmp);
				SolW1Cursor();
				return(Result);
			case F12: return(86); /* three step to main to exit */
/*		 case F1: SolNewGame();
					goto loopback;
*/		 case F9:
					goto loopback;
			case UPARROW:
				if(SolW1y==2) {
					vsync();
					blit(bmp, screen, 0, 0, 0, 0, 800, 600);
					SolW1y=1;
					SolW1x=OrigX;
	rectfill(screen,150+(90*(OrigX-1)),84,150+(90*(OrigX))-1,95+ch,BLACK);
					DrawCard(150+(90*(OrigX-1)),84,PlayerCard);
				destroy_bitmap(bmp);
				SolW1Cursor();
				return(Result);
				} else {
					--SolW1y;
					dy-=sy;
					goto loopback;
				}
			case DOWNARROW:
				if(dy > 500) goto loopback2;
				++SolW1y;
				dy+=sy;
				goto loopback;
			case LEFTARROW:
				if(dx < 170) goto loopback2;
				dx-=sx;
				--Dest;
				goto loopback;
			case RIGHTARROW:
				if(dx > 630) goto loopback2;
				dx+=sx;
				++Dest;
				goto loopback;
			case 350: Screenshot792("Borg8.bmp");
				PanelBeep();
					goto loopback;
			case 351: Screenshot1024("Borg10.bmp");
				PanelBeep();
					goto loopback;
			default: /* ScoldUser(); */
				goto loopback;
		}
cursor:
	SolW1Cursor();
	goto loopback;
}

int FlipHoleCard(int TabNum)
{	/* For convenience, the bottom card is 0, unlike the stock. */

	if(TabDownCount[TabNum]==0) return(1);
	TabUp[TabNum][0]=TabDown [TabNum][TabDownCount[TabNum]-1];
	OldTabDownCount[TabNum]=TabDownCount[TabNum];
	--TabDownCount[TabNum];
	EraseTab(TabNum);
	DrawPregnantTab(TabNum, TabDownCount[TabNum]);
	DrawCard(150+(90*TabNum),264+(TabDownCount[TabNum]*3),TabUp[TabNum][0]);
	TabUpCount[TabNum]=1;
}
int TabToFoundation(int TabNum)
{
	int PlayerCard;

	PlayerCard=TabUp[TabNum][TabUpCount[TabNum]-1];
	if(Foundation[0][FoundationCount[0]]==PlayerCard-1) {
		DrawCard(150+270,84,PlayerCard);
		++FoundationCount[0];
		Foundation[0][FoundationCount[0]]=PlayerCard;
		--TabUpCount[TabNum];
		DrawTab(TabNum);
		UndoFoundationNum=0;
		UndoTabNum=TabNum;
		SolUndoMode=TAB_TO_FOUNDATION;
		SolUndoOn();
		return(0);
	}
	if(Foundation[1][FoundationCount[1]]==PlayerCard-1) {
		DrawCard(150+360,84,PlayerCard);
		++FoundationCount[1];
		Foundation[1][FoundationCount[1]]=PlayerCard;
		--TabUpCount[TabNum];
		DrawTab(TabNum);
		UndoFoundationNum=1;
		UndoTabNum=TabNum;
		SolUndoMode=TAB_TO_FOUNDATION;
		SolUndoOn();
		return(0);
	}
	if(Foundation[2][FoundationCount[2]]==PlayerCard-1) {
		DrawCard(150+450,84,PlayerCard);
		++FoundationCount[2];
		Foundation[2][FoundationCount[2]]=PlayerCard;
		--TabUpCount[TabNum];
		DrawTab(TabNum);
		UndoFoundationNum=2;
		UndoTabNum=TabNum;
		SolUndoMode=TAB_TO_FOUNDATION;
		SolUndoOn();
		return(0);
	}
	if(Foundation[3][FoundationCount[3]]==PlayerCard-1) {
		DrawCard(150+540,84,PlayerCard);
		++FoundationCount[3];
		Foundation[3][FoundationCount[3]]=PlayerCard;
		--TabUpCount[TabNum];
		DrawTab(TabNum);
		UndoFoundationNum=3;
		UndoTabNum=TabNum;
		SolUndoMode=TAB_TO_FOUNDATION;
		SolUndoOn();
		return(0);
	}
	return(1);
}
int WasteToFoundation()
{
	int PlayerCard;

	PlayerCard=Waste[WasteCount-1];
	if(Foundation[0][FoundationCount[0]]==PlayerCard-1) {
		DrawCard(150+270,84,PlayerCard);
		++FoundationCount[0];
		Foundation[0][FoundationCount[0]]=PlayerCard;
		--WasteCount;
		--WasteShowing;
		UpdateWasteDisplay();
		UndoFoundationNum=0;
		SolUndoMode=WASTE_TO_FOUNDATION;
		SolUndoOn();
		return(0);
	}
	if(Foundation[1][FoundationCount[1]]==PlayerCard-1) {
		DrawCard(150+360,84,PlayerCard);
		++FoundationCount[1];
		Foundation[1][FoundationCount[1]]=PlayerCard;
		--WasteCount;
		--WasteShowing;
		UpdateWasteDisplay();
		UndoFoundationNum=1;
		SolUndoMode=WASTE_TO_FOUNDATION;
		SolUndoOn();
		return(0);
	}
	if(Foundation[2][FoundationCount[2]]==PlayerCard-1) {
		DrawCard(150+450,84,PlayerCard);
		++FoundationCount[2];
		Foundation[2][FoundationCount[2]]=PlayerCard;
		--WasteCount;
		--WasteShowing;
		UpdateWasteDisplay();
		UndoFoundationNum=2;
		SolUndoMode=WASTE_TO_FOUNDATION;
		SolUndoOn();
		return(0);
	}
	if(Foundation[3][FoundationCount[3]]==PlayerCard-1) {
		DrawCard(150+540,84,PlayerCard);
		++FoundationCount[3];
		Foundation[3][FoundationCount[3]]=PlayerCard;
		--WasteCount;
		--WasteShowing;
		UpdateWasteDisplay();
		UndoFoundationNum=3;
		SolUndoMode=WASTE_TO_FOUNDATION;
		SolUndoOn();
		return(0);
	}
	return(1);
}
int SolGiveCards()
{
	int z;

	/* for Undo */
	for(z=0; z < 24; ++z) PrevSolStock[z]=SolStock[z];
	PrevSolStockCount=SolStockCount;
	PrevSolStockUndealt =SolStockUndealt;
	for(z=0; z < 24; ++z) PrevWaste[z]=Waste[z];
	PrevWasteCount=WasteCount;
	PrevWasteShowing=WasteShowing;
	SolUndoMode=DEAL;
	SolUndoOn();
	if(SolStockUndealt >= 3) {
		for(z=0; z < 3; ++z) {
			Waste[WasteCount]=SolStock[z];
			DrawCard(150+90+(z*14),84,Waste[WasteCount]);
			++WasteCount;
		}
		for(z=0; z < SolStockUndealt-3; ++z) {
			SolStock[z]=SolStock[z+3];
		}
		SolStockCount -=3;
		SolStockUndealt -=3;
		if(SolStockCount > 0) { /* animation */
			Animate=1;
			DrawSolStock();	/* show thinner deck */
		}
		if(SolStockUndealt==0) SolDeckEmpty();
		WasteShowing=3;
		return(0);
	}
	if(SolStockUndealt==2) {
		rectfill(screen,240,80,272+cw,90+ch,BLACK); /* erase waste */
		for(z=0; z < 2; ++z) {
			Waste[WasteCount]=SolStock[z];
			DrawCard(150+90+(z*14),84,Waste[WasteCount]);
			++WasteCount;
		}
		SolStockCount=0;
		SolStockUndealt=0;
		WasteShowing=2;
		SolDeckEmpty();
		return(0);
	}
	if(SolStockUndealt==1) {
		rectfill(screen,240,80,272+cw,90+ch,BLACK); /* erase waste */
		for(z=0; z < 1; ++z) {
			Waste[WasteCount]=SolStock[z];
			DrawCard(150+90+(z*14),84,Waste[WasteCount]);
			++WasteCount;
		}
		SolStockCount=0;
		SolStockUndealt=0;
		WasteShowing=1;
		SolDeckEmpty();
		return(0);
	}
	if(SolStockUndealt==0) {
		DrawSolStock();
		for(z=0; z < WasteCount; ++z) {
			SolStock[z]=Waste[z];
		}
		SolStockCount=WasteCount;
		SolStockUndealt=SolStockCount;
		WasteCount=0;
		WasteShowing=0;
		if(SolStockCount > 0) DrawSolStock();
		rectfill(screen,240,80,272+cw,90+ch,BLACK); /* erase waste */
	}
}
int UpdateWasteDisplay()
{
	rectfill(screen,240,80,272+cw,90+ch,BLACK); /* erase waste */
	if(WasteCount==0) return(0);
	if(WasteShowing==1) {
		DrawCard(150+90,84,Waste[WasteCount-1]);
	}
	if(WasteShowing==2) {
		DrawCard(150+90,84,Waste[WasteCount-2]);
		DrawCard(150+90+14,84,Waste[WasteCount-1]);
	}
	if(WasteShowing==3) {
		DrawCard(150+90,84,Waste[WasteCount-3]);
		DrawCard(150+90+14,84,Waste[WasteCount-2]);
		DrawCard(150+90+28,84,Waste[WasteCount-1]);
	}
/* show waste a card at a time if the first three are removed */
	if(WasteShowing==0 && WasteCount > 0) {
		DrawCard(150+90,84,Waste[WasteCount-1]);
		WasteShowing=1;
	}
}
int SolDeckEmpty()
{
	rectfill(screen,140,80,150+cw,90+ch,BLACK); /* erase stock */
	DrawSpot(150,84);
}
int SolNewGame()
{
	int a,b,z;
	unsigned int x;

	if(RedoSignage==OK) {
		rectfill(screen,10,308,110,532,BLACK);
		SolBoxes();
	}
	RedoSignage=NG;

	for(z=0; z<52; ++z) SolDeck[z]=z+1;
	shuffle(SolDeck, 52);

	for(z=0; z<24; ++z) SolStock[z]=SolDeck[z];
	SolStockCount=24;
	for(z=24; z<31; ++z) {
		TabUp[z-24][0]=SolDeck[z];
		TabUpCount[z-24]=1;
	}
/* deal the tableau hole cards the lazy, chicken but safe way */
	for(z=0; z<7; ++z) TabDownCount[z]=z;
	TabDown[1][0]=SolDeck[31];

	TabDown[2][0]=SolDeck[32];
	TabDown[2][1]=SolDeck[33];

	TabDown[3][0]=SolDeck[34];
	TabDown[3][1]=SolDeck[35];
	TabDown[3][2]=SolDeck[36];

	TabDown[4][0]=SolDeck[37];
	TabDown[4][1]=SolDeck[38];
	TabDown[4][2]=SolDeck[39];
	TabDown[4][3]=SolDeck[40];

	TabDown[5][0]=SolDeck[41];
	TabDown[5][1]=SolDeck[42];
	TabDown[5][2]=SolDeck[43];
	TabDown[5][3]=SolDeck[44];
	TabDown[5][4]=SolDeck[45];

	TabDown[6][0]=SolDeck[46];
	TabDown[6][1]=SolDeck[47];
	TabDown[6][2]=SolDeck[48];
	TabDown[6][3]=SolDeck[49];
	TabDown[6][4]=SolDeck[50];
	TabDown[6][5]=SolDeck[51]; /* the last card */

	for(z=0; z<4; ++z) FoundationCount[z]=0;
	Foundation[0][0]=0;  /* 1 lower than its ace,			*/
	Foundation[1][0]=13; /* to be incremented as it fills up	 */
	Foundation[2][0]=26;
	Foundation[3][0]=39;
	SolStockUndealt=24;
	WasteCount=0;
	rectfill(screen,240,80,272+cw,90+ch,BLACK); /* erase waste */
	rectfill(screen, 130, 254, 790, 590, BLACK); /* erase all tableaus*/
	SolW1x=1;
	SolW1y=1;
	SolUndoMode=NG;
	SolUndoOff();
	SolW1Cursor();
	SolDisplayNew();
	SolGameOver=0;
}
int SolDisplayNew()
{
	int a,b;

	for(a=0; a<7; ++a) {
		DrawPregnantTab(a,a);
		DrawCard(150+(90*a),264+(a*3),TabUp[a][0]);
	}

/* top */

	DrawSolStock();
	rectfill(screen,420, 84, 780, 184, BLACK); /* waste gone */
	DrawSpot(150+270,84);	/* empty foundations */
	DrawSpot(150+360,84);
	DrawSpot(150+450,84);
	DrawSpot(150+540,84);
}
int DrawSolStock()
{
	if(SolStockCount + WasteCount==0) {
		rectfill(screen,140,80,150+cw,90+ch,BLACK); /* erase stock */
		DrawSpot(150,84);
		return(1);
	}
	if(SolStockCount < 3)	rectfill(screen,140,80,150+cw,90+ch,BLACK);
/* last argument 0=full card back not necessary */
	if(SolStockCount > 2) DrawDownCard(146,82,0);
	if(SolStockCount > 1)	DrawDownCard(148,83,0);
	DrawDownCard(150,84,1);
}
int UndoDeal()
{	 /* Undo a deal of up to 3 cards from stock to waste */
	int z;

	for(z=0; z < 24; ++z) SolStock[z]=PrevSolStock[z];
	SolStockCount=PrevSolStockCount;
	SolStockUndealt=PrevSolStockUndealt;
	for(z=0; z < 24; ++z) Waste[z]=PrevWaste[z];
	WasteCount=PrevWasteCount;
	WasteShowing=PrevWasteShowing;
	SolUndoMode=NG;
	SolUndoOff();
	DrawSolStock();
	if(SolStockUndealt==0) SolDeckEmpty();
	UpdateWasteDisplay();
}
int UndoWasteToTab()
{
	++WasteCount;
	++WasteShowing;
	UpdateWasteDisplay();
	--TabUpCount[UndoTabNum];
	EraseTab(UndoTabNum);
	DrawTab(UndoTabNum);
	SolW1x=2;
	SolW1y=1;
	SolW1Cursor();
	SolUndoMode=NG;
	SolUndoOff();
}
int UndoWasteToFoundation()
{
	int PlayerCard;
	int OrigX=UndoFoundationNum + 4; /* to enable copied pasted code */

	++WasteCount;
	++WasteShowing;
	UpdateWasteDisplay();
	PlayerCard=Foundation[UndoFoundationNum][FoundationCount[UndoFoundationNum]];

	if(FoundationCount[UndoFoundationNum]==1) {
	rectfill(screen,150+(90*(OrigX-1)),84,150+(90*(OrigX))-1,95+ch,BLACK);
		DrawSpot(150+(90*(OrigX-1)),84);
	} else {
	rectfill(screen,120+(90*(OrigX)),84,150+(90*(OrigX))-1,95+ch,BLACK);
		DrawCard(150+(90*(OrigX-1)),84,PlayerCard-1);
	}
	--FoundationCount[UndoFoundationNum];
	SolW1x=2;
	SolW1y=1;
	SolW1Cursor();
	SolUndoMode=NG;
	SolUndoOff();
}
int UndoTabToFoundation()
{
	int PlayerCard;
	int OrigX=UndoFoundationNum + 4; /* to enable copied pasted code */

	++TabUpCount[UndoTabNum];
	DrawTab(UndoTabNum);
	PlayerCard=Foundation[UndoFoundationNum][FoundationCount[UndoFoundationNum]];
	if(FoundationCount[UndoFoundationNum]==1) {
	rectfill(screen,150+(90*(OrigX-1)),84,150+(90*(OrigX))-1,95+ch,BLACK);
		DrawSpot(150+(90*(OrigX-1)),84);
	} else {
		DrawCard(150+(90*(OrigX-1)),84,PlayerCard-1);
	}
	--FoundationCount[UndoFoundationNum];
	SolW1Cursor();
	SolUndoMode=NG;
	SolUndoOff();
}
void UndoFoundationToTab()
{
	int PlayerCard;
	int OrigX=UndoFoundationNum + 4; /* to enable copied pasted code */

	++FoundationCount[UndoFoundationNum];
	--TabUpCount[UndoTabNum];
	DrawTab(UndoTabNum);
	PlayerCard=Foundation[UndoFoundationNum][FoundationCount[UndoFoundationNum]];
	DrawCard(150+(90*(OrigX-1)),84,PlayerCard);
	SolW1Cursor();
	SolUndoMode=NG;
	SolUndoOff();
}
void UndoTabToTab()
{
	TabUpCount[UndoTabNum] -=UndoLength ;
	TabUpCount[UndoSourceTabNum] += UndoLength;
	EraseTab(UndoTabNum);
	DrawTab(UndoTabNum);
	DrawTab(UndoSourceTabNum);
	SolW1Cursor();
	SolUndoMode=NG;
	SolUndoOff();
}
int SolUndo()
{
	if(SolUndoMode==NG) return(1);
	if(SolUndoMode==DEAL) UndoDeal();
	if(SolUndoMode==WASTE_TO_TAB) UndoWasteToTab();
	if(SolUndoMode==WASTE_TO_FOUNDATION) UndoWasteToFoundation();
	if(SolUndoMode==TAB_TO_FOUNDATION) UndoTabToFoundation();
	if(SolUndoMode==FOUNDATION_TO_TAB) UndoFoundationToTab();
	if(SolUndoMode==TAB_TO_TAB) UndoTabToTab();
	return(0);
}
int SolGameWon()
{ /* a meager visual reward for winning */
	int down=314;

	MarsF(150,460-72,3, BRICK,BLACK,"RESISTANCE IS VITAL!");
	PutsLuna(150,460-72+88, LAVENDER,BLACK, "WWW.STARTREK.COM");
	set_palette(faces_palette);
	blit(faces, screen, 0, 0, 150, 380-72, 565, 77);
	SolGameOver=1;
	/* kill most bracket signage until new game */
	rectfill(screen,10,105,110,122,PALEBLUE);
	rectfill(screen,10,130,110,150,MANGO);
	rectfill(screen,10,308,110,532,BRICK);

	PutsLuna(14,down, BLACK,BRICK, "ENS. KIM");
	down += 18;
	PutsLuna(14,down, BLACK,BRICK, "EMH");
	down += 18;
	PutsLuna(14,down, BLACK,BRICK, "LT. BARKLAY");
	down += 18;
	PutsLuna(14,down, BLACK,BRICK, "AMB. NEELIX");
	down += 18;
	PutsLuna(14,down, BLACK,BRICK, "CPT. JANEWAY");
	down += 18;
	PutsLuna(14,down, BLACK,BRICK, "LT. TORRES");
	down += 18;
	PutsLuna(14,down, BLACK,BRICK, "BORG QUEEN");
	down += 18;
	PutsLuna(14,down, BLACK,BRICK, "SEVEN OF NINE");
	down += 18;
	PutsLuna(14,down, BLACK,BRICK, "CMDR. CHAKOTAY");
	down += 18;
	PutsLuna(14,down, BLACK,BRICK, "CMDR. TUVOK");
	down += 18;
	PutsLuna(14,down, BLACK,BRICK, "Q");
	down += 18;
	PutsLuna(14,down, BLACK,BRICK, "LT. PARIS");
	RedoSignage=OK;
}
int SolitaireWindowActive()
{
	int Result,a,z,Checksum,Count,Count2,Total;

	SolW1Cursor();
loopback:

Checksum=0;
for(z=0; z < SolStockCount; ++z) Checksum += SolStock[z];
for(z=0; z < WasteCount; ++z) Checksum += Waste[z];

for(z=1; z <= FoundationCount[0]; ++z) Checksum += Foundation[0][z];
for(z=1; z <= FoundationCount[1]; ++z) Checksum += Foundation[1][z];
for(z=1; z <= FoundationCount[2]; ++z) Checksum += Foundation[2][z];
for(z=1; z <= FoundationCount[3]; ++z) Checksum += Foundation[3][z];

for(z=0; z < TabDownCount[0]; ++z) Checksum += TabDown[0][z];
for(z=0; z < TabDownCount[1]; ++z) Checksum += TabDown[1][z];
for(z=0; z < TabDownCount[2]; ++z) Checksum += TabDown[2][z];
for(z=0; z < TabDownCount[3]; ++z) Checksum += TabDown[3][z];
for(z=0; z < TabDownCount[4]; ++z) Checksum += TabDown[4][z];
for(z=0; z < TabDownCount[5]; ++z) Checksum += TabDown[5][z];
for(z=0; z < TabDownCount[6]; ++z) Checksum += TabDown[6][z];

for(z=0; z < TabUpCount[0]; ++z) Checksum += TabUp[0][z];
for(z=0; z < TabUpCount[1]; ++z) Checksum += TabUp[1][z];
for(z=0; z < TabUpCount[2]; ++z) Checksum += TabUp[2][z];
for(z=0; z < TabUpCount[3]; ++z) Checksum += TabUp[3][z];
for(z=0; z < TabUpCount[4]; ++z) Checksum += TabUp[4][z];
for(z=0; z < TabUpCount[5]; ++z) Checksum += TabUp[5][z];
for(z=0; z < TabUpCount[6]; ++z) Checksum += TabUp[6][z];


Count=WasteCount + SolStockCount;
Count2=FoundationCount[0];
Count2 += FoundationCount[1];
Count2 += FoundationCount[2];
Count2 += FoundationCount[3];

if(Count2==52) SolGameWon();

loopback2:

if(Checksum !=1378) {
	MarsF(150,10,2, MANGO,BLACK,"ERROR");
	ScoldUser();
}
	do {
		SolPutTime();
	} while(!kbhit());
	c=GetKey();
	switch(c)
		{

			case 8:	if(SolGameOver==1) goto loopback2;
					SolUndo();
					goto loopback;
			case 13: if(SolGameOver==1) goto loopback2;
					if(SolW1y==1 && SolW1x==1) {
						SolGiveCards();
						goto loopback;
					}
					if(SolW1y > 1 && TabUpCount[SolW1x-1]==0) {
						if(TabDownCount[SolW1x-1] > 0) {
							FlipHoleCard(SolW1x-1);
							SolW1Cursor();
							SolUndoMode=NG;
							SolUndoOff();
						}
						goto loopback;
					}
					if(SolW1y > 1) {
						Result=DragFromTab();
						if(Result==86) return(0); /* exit program */
					}
					if(SolW1y==1 && SolW1x==2) {
						Result=DragFromWaste();
						if(Result==86) return(0); /* exit program */
						goto loopback;
					}
					if(SolW1y==1 && SolW1x > 3) {
						Result=DragFromFoundation();
						if(Result==86) return(0); /* exit program */
						goto loopback;
					}
					goto loopback;
		case 32:
		case 397:
		case PGUPKEY: if(SolGameOver==1) goto loopback2;
					do {
						Result=0;
						Result=WasteToFoundation();
						for(z=0; z<7; ++z)
							Result=Result + TabToFoundation(z);
					} while(Result < 8);
				SolW1Cursor();
				goto loopback;
		case 372:
		case PGDNKEY: if(SolGameOver==1) goto loopback2;
				if(SolW1y > 1) TabToFoundation(SolW1x-1);
					else WasteToFoundation();
				SolW1Cursor();
				goto loopback;
			case F12: return(0); /* two steps to main to exit */
			case F1: SolNewGame();
				goto loopback;
			case F11: Sol24Help();
				PanelBeep();
				goto loopback;
			case F9:
				goto loopback;
			case F6: /* DrawTab(SolW1x-1); */
				goto loopback;
			case UPARROW: if(SolGameOver==1) goto loopback2;
				if(SolW1y==1) goto loopback;
				--SolW1y;
				if(SolW1y==1) SolW1x=1;
				SolW1Cursor();
				goto loopback;
			case DOWNARROW: if(SolGameOver==1) goto loopback2;
				if(SolW1y > 1 && SolW1y >= TabUpCount[SolW1x-1] + 1) goto loopback;
				++SolW1y;
				SolW1Cursor();
				goto loopback;
			case LEFTARROW: if(SolGameOver==1) goto loopback2;
				if(SolW1y > 2) SolW1y=2;
				if(SolW1x>1) --SolW1x;
				SolW1Cursor();
				goto loopback;
			case RIGHTARROW: if(SolGameOver==1) goto loopback2;
				if(SolW1y > 2) SolW1y=2;
				if(SolW1x < 7) ++SolW1x;
				SolW1Cursor();
				goto loopback;
			case 350: Screenshot792("Borg8.bmp");
				PanelBeep();
				goto loopback;
			case 351: Screenshot1024("Borg10.bmp");
				PanelBeep();
				goto loopback;
			default: ScoldUser();
				goto loopback;
		}
cursor:
	SolW1Cursor();
	goto loopback;
}
void SolInit()
{
	min_old=70;
	OldSolW1x=3;
	OldSolW1y=2;
	SolW1x=1;
	SolW1y=1;
	SolGameOver=0;
	RedoSignage=NG;
	cw=71;  /* card width */
	ch=100; /* card height */
	Animate = 0;
}
void SolStartScreen()
{
	rectfill(screen,0,0,800,600,BLACK);
	Brackets1(MANGO,PALEBLUE);
	SolBoxes();
	MarsF(780-(StrLenScorpio("BORG SOLITAIRE")*2),10,2, BORG,BLACK,"BORG");
	MarsF(780-(StrLenScorpio("SOLITAIRE")*2),10,2, ORANGE,BLACK,"SOLITAIRE");
}
int SolitaireMode()
{
	BeepSet=TRUE;
	SolInit();
	SolStartScreen();
	strcpy(CurrentFile, "C:\\LCARS24\\LC24APPS\\SOL24\\FACES.BMP");
	faces=load_bitmap(CurrentFile, faces_palette);
	if (!faces) {
		rest(1000);
		NormalLoadError(CurrentFile);
		return(86);
	}
	SolNewGame();
	SolitaireWindowActive();
	SolStartScreen();
	DrawSpot(150+270,84);	/* empty foundations */
	DrawSpot(150+360,84);
	DrawSpot(150+450,84);
	DrawSpot(150+540,84);
	DelayedShutdown();
	destroy_bitmap(faces);
}
/* eof */

