/* LFRAC.C -- Fraction Commands

	Written April 1994 by Craig A. Finseth
	Copyright 1994 by Craig A. Finseth
*/

#include "loki.h"
#if defined(FLOAT)
double modf();
#endif

static char fbuf[SMALLBUFFSIZE];

void F_Fixed(int *num, int *denom, enum FRACTYPE *ft, struct number *n);
void F_Most(int *num, int *denom, enum FRACTYPE *ft, struct number *n);
int F_GCD(int a, int b);

/* ------------------------------------------------------------ */

/* Create a fraction from the number according to the current fraction
mode.. */

void
FDo(int *numerator, int *denominator, enum FRACTYPE *ft, struct number *n)
	{
	int gcd;

	if (m.frac_mode == 1) {		/* fixed */
		F_Fixed(numerator, denominator, ft, n);
		}
	else if (m.frac_mode == 2) {	/* factors of */
		F_Fixed(numerator, denominator, ft, n);
		gcd = F_GCD(*numerator, *denominator);
		*numerator /= gcd;
		*denominator /= gcd;
		}
	else if (m.frac_mode == 3) {	/* most precise */
		F_Most(numerator, denominator, ft, n);
		}
	else	{
		*numerator = 1;
		*denominator = 1;
		*ft = FT_EXACT;
		}
	}


/* ------------------------------------------------------------ */

/* Format a fraction into a static buffer. */

char *
FFormat(struct number *n)
	{
	int nu;
	int d;
	enum FRACTYPE ft;
	struct number nn = *n;

#if defined(FLOAT)
	if (nn.f < 0.0) nn.f = -nn.f;
#endif
#if defined(BCD)
	if (BIsNeg(&nn)) BNeg(&nn);
#endif
	FDo(&nu, &d, &ft, &nn);
	xsprintf(fbuf, " %d/%d%c ", nu, d,
		Res_String(NULL, RES_CONF, RES_FRACMARKS)[(int) ft]);
	if (m.frac_mode == 0 ||
		 (m.frac_mode > 1 && nu == 0 && ft == FT_EXACT)) {
		return("");
		}
	return(fbuf);
	}


/* ------------------------------------------------------------ */

/* Set the fraction mode. */

void
FMode(void)
	{
	if (isuarg) {
		m.frac_mode = uarg;
		}
	else	{
		m.frac_mode++;
		}
	if (m.frac_mode > 3) m.frac_mode = 0;
	DChangedView();
	DChanged(US_BOTTOMBAR);
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Set the fraction denominator. */

void
FSetDenom(void)
	{
	if (!isuarg) {
		DError(RES_EXPLARG);
		}
	else	{
		m.frac_denom = max(min(uarg, 10000), 2);
		DChangedView();
		DChanged(US_BOTTOMBAR);
		}
	uarg = 0;
	}


/* ------------------------------------------------------------ */

/* Compute the fraction for a fixed denominator. */

void
F_Fixed(int *num, int *denom, enum FRACTYPE *ft, struct number *n)
	{
#if defined(FLOAT)
	double fint;
	double ffrac = modf(n->f * m.frac_denom, &fint);

	if (n->f == 0.0) {
		*num = 0;
		*denom = m.frac_denom;
		*ft = FT_EXACT;
		return;
		}

/* now have numerator = fint, denom = m.frac_denom, diff = ffrac */

	*num = fint;
	*denom = m.frac_denom;
	*ft = FT_EXACT;

	if (ffrac == 0.0) return;

/* question: is (numerator + 1) / denom closer? which is to say:

	A		B		C	

	n / d		fraction	n+1 / d

which is smaller, B-A (which is ffrac) or C-B? */

	if (ffrac < 0.5) {
#endif
#if defined(BCD)
	struct number fint;
	struct number ffrac;
	struct number frac_denom;
	struct number half;

	if (BIsZero(n)) {
		*num = 0;
		*denom = m.frac_denom;
		*ft = FT_EXACT;
		return;
		}

	fint = *n;
	BAssignI(&frac_denom, m.frac_denom);
	BMul(&fint, &frac_denom);
	BModf(&fint, &ffrac);

/* now have numerator = fint, denom = m.frac_denom, diff = ffrac */

	*num = BLong(&fint);
	*denom = m.frac_denom;
	*ft = FT_EXACT;

	if (BIsZero(&ffrac)) return;

/* question: is (numerator + 1) / denom closer? which is to say:

	A		B		C	

	n / d		fraction	n+1 / d

which is smaller, B-A (which is ffrac) or C-B? */

	BEnter(&half, ".5");
	BSub(&ffrac, &half);
	if (BIsNeg(&ffrac)) {
#endif
		/* no */
		*ft = FT_TOOLOW;
		}
	else	{	/* yes */
		(*num)++;
		*ft = FT_TOOHIGH;
		}
	}


/* ------------------------------------------------------------ */

/* Compute the fraction for a most precise fraction. */

void
F_Most(int *num, int *denom, enum FRACTYPE *ft, struct number *n)
	{
	int cnt;
#if defined(FLOAT)
	double fint;
	double ffrac;
	double faccuracy = 100.0;

	if (n->f == 0.0) {
		*num = 0;
		*denom = m.frac_denom;
		*ft = FT_EXACT;
		return;
		}

	*num = 0;
	*denom = 2;
	*ft = FT_EXACT;
	for (cnt = 2; cnt <= m.frac_denom; cnt++) {
		ffrac = modf(n->f * cnt, &fint);
		if (ffrac == 0.0) {
			*num = fint;
			*denom = cnt;
			*ft = FT_EXACT;
			return;
			}
		if (ffrac < faccuracy || 1.0 - ffrac < faccuracy) {
			*num = fint;
			*denom = cnt;
			if (ffrac < 0.5) {
				*ft = FT_TOOLOW;
				}
			else	{
				(*num)++;
				*ft = FT_TOOHIGH;
				}
			faccuracy = (ffrac < faccuracy) ? ffrac : 1.0 - ffrac;
			}
		}
#endif
#if defined(BCD)
	struct number fint;
	struct number ffrac;
	struct number faccuracy;
	struct number tmp;
	struct number ncnt;
	struct number half;
	struct number one;

	if (BIsZero(n)) {
		*num = 0;
		*denom = m.frac_denom;
		*ft = FT_EXACT;
		return;
		}

	BAssignI(&faccuracy, 100);

	*num = 0;
	*denom = 2;
	*ft = FT_EXACT;
	BEnter(&half, ".5");
	BAssign(&one, 1);

	for (cnt = 2; cnt <= m.frac_denom; cnt++) {
		fint = *n;
		BAssignI(&ncnt, cnt);
		BMul(&fint, &ncnt);
		BModf(&fint, &ffrac);

		if (BIsZero(&ffrac)) {
			*num = BLong(&fint);
			*denom = cnt;
			*ft = FT_EXACT;
			return;
			}
		tmp = ffrac;
		BSub(&tmp, &faccuracy);
		if (BIsNeg(&tmp)) {
			*num = BLong(&fint);
			*denom = cnt;
			faccuracy = ffrac;
			BSub(&ffrac, &half);
			if (BIsNeg(&ffrac)) {
				*ft = FT_TOOLOW;
				}
			else	{
				(*num)++;
				*ft = FT_TOOHIGH;
				}
			continue;
			}

		tmp = one;
		BSub(&tmp, &ffrac);
		BSub(&tmp, &faccuracy);
		if (BIsNeg(&tmp)) {
			*num = BLong(&fint);
			*denom = cnt;
			faccuracy = one;
			BSub(&faccuracy, &ffrac);

			BSub(&ffrac, &half);
			if (BIsNeg(&ffrac)) {
				*ft = FT_TOOLOW;
				}
			else	{
				(*num)++;
				*ft = FT_TOOHIGH;
				}
			}
		}
#endif
	}


/* ------------------------------------------------------------ */

/* Compute and return the greatest common denominator.  A, B must both
be >= 0. */

int
F_GCD(int a, int b)
	{
	int x;
	int y;

	if (a == b) return(a);
	if (a == 0 || b == 0) return(1);
	if (b > a) return(F_GCD(b, a));

	/* we know that a > b */
	for (;;) {
		x = a % b;
		if (x == 0) return(b);
		y = b / x;
		if (b == y * x) return(x);
		a = b;
		b = x;
		}
	}


/* end of LFRAC.C -- Fraction Commands */
