library hp200lx;

{
***** version 1.2 ***** 
(c) Pavel Zampach (zampach@volny.cz), 2005
GNU GENERAL PUBLIC LICENSE 
Source code for Borland Delphi Compiler (originally version 7.0)
}

{$E wfx}
{$IOChecks Off}
{$ImageBase $40008000}
{$OverflowChecks On}
{$RangeChecks On}
{$Resource rc\HP200LX.RES}

{
  This project uses Key Objects Library
  KOL is provided free with the source code.
  Copyright (C) Vladimir Kladov, 2000-2003.
  Web-Page: http://bonanzas.rinet.ru
}

uses
  Windows, SysUtils, KOL,
  SetForm  in 'SETFORM.PAS',
  Fsplugin in 'FSPLUGIN.PAS',
  Filer    in 'FILER.PAS',
  DirCache in 'DIRCACHE.PAS';

const
  PLUGIN_TITLE = 'HPLX file system';
  HELLO_TITLE  = 'HPLX plugin 1.2/GNU GPL, (c) P. Zampach, 2005';
  DIR_HPLX     = 0;
  DIR_CACHE    = 1;
  DIR_ROOT     = 2;

var
  ProgressProc : tProgressProc;
  LogProc      : tLogProc;
  RequestProc  : tRequestProc;
  PluginNumber, DiskNr : integer;
  SDrives, IniFileName : string;


{ ------------------------------------------------------------------ }

procedure dirEntryToFindData (const DirEntry : TDirEntry; var FindData : tWIN32FINDDATA);

var
  FT : TFileTime;
  
begin
  fillChar (FindData, sizeOf (tWIN32FINDDATA), 0);

  FindData.dwFileAttributes := DirEntry.fattr;
  DosDateTimeToFileTime (DirEntry.fdate, DirEntry.ftime, FT);
  LocalFileTimeToFileTime (FT, FindData.ftLastWriteTime);
  FindData.nFileSizeLow := DirEntry.fsize;
  strPLCopy (FindData.cFileName, DirEntry.fname, FNAME_LENGTH);
end;

{ ------------------------------------------------------------------ }

procedure rootToFindData (var FindData : tWIN32FINDDATA);

begin
  fillChar (FindData, sizeOf (tWIN32FINDDATA), 0);

  FindData.dwFileAttributes := FILE_ATTRIBUTE_DIRECTORY;
  FindData.ftLastWriteTime.dwHighDateTime := $FFFFFFFF;
  FindData.ftLastWriteTime.dwLowDateTime  := $FFFFFFFE;
  strPLCopy (FindData.cFileName, SDrives[DiskNr] + ':', 2);
end;

{ ------------------------------------------------------------------ }

function cacheAdd (const RemoteName : string) : boolean;

var
  DirEntry : TDirEntry;

begin
  HPLXSearchDirName (RemoteName);
  Result := HPLXSearchDir (DirEntry);
  if HPLXError then begin
    Result := false;
    filerReset;
  end;
  
  if Result then
    cacheWrite (extractfilePath (RemoteName), DirEntry);  
end;

{ ------------------------------------------------------------------ }

function HPLXConnect : boolean;

var
  CommPort  : string;
  CommSpeed : integer;
  Ini       : PIniFile;
  Dsk       : char;

begin
  Result := true;
  if Connected then exit;
  
  Ini := OpenIniFile (IniFileName);
  Ini.Section := 'HPLX';
  Ini.Mode  := ifmRead;
  CommPort  := Ini.ValueString ('Port', '');
  CommSpeed := Ini.ValueInteger ('Speed', -1);
  SDrives   := Ini.ValueString ('Drives', '');
  Ini.Free;
  LogProc (PluginNumber, MsgTYPE_CONNECT, 'CONNECT \');
  
  if CommSpeed = -1 then begin
    Result := false;
    SetLastError (ERROR_FILE_NOT_FOUND);
    LogProc (PluginNumber, MsgTYPE_IMPORTANTERROR, 'FSPLUGIN.INI not found');
    exit;
  end;
  
  if not filerConnect (CommPort, CommSpeed) then begin
    Result := false;
    SetLastError (ERROR_FILE_NOT_FOUND);
    LogProc (PluginNumber, MsgTYPE_IMPORTANTERROR, 'Connect failed');
    exit;
  end;
  
  cacheInit;
  LogProc (PluginNumber, MsgTYPE_OPERATIONCOMPLETE, 'HPLX connected');
  if SDrives = '' then begin
    LogProc (PluginNumber, MsgTYPE_DETAILS, 'Autodetect running...');
    for Dsk := 'A' to 'Z' do begin
      if HPLXFileExists (Dsk + ':\') then
        SDrives := SDrives + Dsk;
      if HPLXError then filerReset
    end;      
  end;
  LogProc (PluginNumber, MsgTYPE_DETAILS, HELLO_TITLE);
end;

{ ****************************************************************** }

function FsInit (PluginNr : integer; pProgressProc : tProgressProc; pLogProc : tLogProc;
                pRequestProc : tRequestProc) : integer; stdcall;

begin
  ProgressProc := pProgressProc;
  LogProc      := pLogProc;
  RequestProc  := pRequestProc;
  PluginNumber := PluginNr;

  Result := 0;
end;

{ ------------------------------------------------------------------ }

function FsFindFirst (Path : PChar; var FindData : tWIN32FINDDATA) : thandle; stdcall;

var
  DirEntry  : TDirEntry;
  DirFlag   : boolean;
  ActualDir : string;
  
begin
  if not HPLXConnect then begin             // Login
    Result := INVALID_HANDLE_VALUE;
    exit;
  end;  

  if Path = '\' then begin                  // Root processing
    Result := DIR_ROOT;
    DiskNr := 1;
    rootToFindData (FindData);
    exit;
  end;

{$Warnings Off}
  ActualDir := IncludeTrailingBackslash (Path + 1); // Skip over leading backslash
{$Warnings On}
  
  if isInCache (ActualDir) then begin       // reading dir from cache
    Result := DIR_CACHE;
    cacheReset (ActualDir);
    if not cacheRead (DirEntry) then begin
      SetLastError (ERROR_NO_MORE_FILES);
      Result := INVALID_HANDLE_VALUE;
      exit;
    end;
  end
  else begin                                // reading dir from HPLX
    Result := DIR_HPLX;
    HPLXSearchDirName (ActualDir + '*.*');
    DirFlag := HPLXSearchDir (DirEntry);
    if HPLXError  then begin
      filerReset;
      SetLastError (ERROR_FILE_NOT_FOUND);
      Result := INVALID_HANDLE_VALUE;
      exit;
    end;
    if not DirFlag then begin
      SetLastError (ERROR_NO_MORE_FILES);
      Result := INVALID_HANDLE_VALUE;
      exit;
    end;
    cacheWrite (ActualDir, DirEntry);
  end;  

  dirEntryToFindData (DirEntry, FindData);
end;                                        // function FsFindFirst

{ ------------------------------------------------------------------ }

function FsFindNext (Hdl : thandle; var FindData : tWIN32FINDDATA) : bool; stdcall;

var
  DirEntry : TDirEntry;

begin
  Result := false;

  case Hdl of
    DIR_HPLX :
    begin
      Result := HPLXSearchDir (DirEntry);
      if HPLXError then begin
        Result := false;
        filerReset;
        exit;
      end;
      if Result then begin
        cacheWrite ('', DirEntry);
        dirEntryToFindData (DirEntry, FindData);
      end;  
    end;  
    
    DIR_CACHE :
    begin
      Result := cacheRead (DirEntry);
      if Result then
        dirEntryToFindData (DirEntry, FindData);
    end;
    
    DIR_ROOT :
    begin
      Result := (DiskNr < length (SDrives));
      if Result then begin
        inc (DiskNr);
        rootToFindData (FindData);
      end;
    end;
  end;                                      // case
end;                                        // function FsFindNext

{ ------------------------------------------------------------------ }

function FsFindClose (Hdl : thandle) : integer; stdcall;

begin
  Result := 0;
end;

{ ------------------------------------------------------------------ }

function FsDisconnect (DisconnectRoot : PChar) : bool; stdcall;

begin
  Result := filerDisconnect;
end;

{ ------------------------------------------------------------------ }

function FsDeleteFile (RemoteName : PChar) : bool; stdcall;

begin
  inc (RemoteName);                         // skip over leading backslash
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('Del ' + RemoteName));

  HPLXDeleteFile (RemoteName);
  if HPLXError then begin;
    Result := false;
    filerReset;
    exit;
  end;

  Result := true;
  cacheDelete (RemoteName);
end;                                        // function FsDeleteFile

{ ------------------------------------------------------------------ }

function FsMkDir (RemoteDir : PChar) : bool; stdcall;

begin
  inc (RemoteDir);                          // skip over leading backslash
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('MkDir ' + RemoteDir));
  Result := false;

  HPLXMkDir (RemoteDir);
  if HPLXError then begin
    filerReset;
    exit;
  end;

  if isInCache (extractFilePath (RemoteDir)) then
    if not cacheAdd (RemoteDir) then exit;

  Result := true;  
end;                                        // function FsMkDir

{ ------------------------------------------------------------------ }

function FsRemoveDir (RemoteName : PChar) : bool; stdcall;

begin
  inc (RemoteName);                         // skip over leading backslash
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('DelDir ' + RemoteName));

  HPLXRmDir (RemoteName);
  if HPLXError then begin
    Result := false;
    filerReset;
    exit;
  end;

  Result := true;
  cacheDelete (RemoteName);
end;                                        // function FsRemoveDir

{ ------------------------------------------------------------------ }

function FsSetTime (RemoteName : PChar; CreationTime, LastAccessTime,
                    LastWriteTime : PFileTime) : bool; stdcall;

var
  LFT   : TFileTime;
  DosDT : TDOSFileTime;

begin  
  if LastWriteTime = nil then begin
    Result := true;
    exit;
  end;  

  inc (RemoteName);                         // skip over leading backslash
  Result := false;
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('Set FileTime ' + RemoteName));

  fileTimeToLocalFileTime (LastWriteTime^, LFT);
  fileTimeToDosDateTime (LFT, DosDT.fdate, DOSDT.ftime);
  HPLXReset (RemoteName);
  HPLXSetFileTime (longword (DosDT));
  HPLXCloseFile;
  if HPLXError then begin
    filerReset;
    exit;
  end;

  if isInCache (extractFilePath (RemoteName)) then begin
    cacheDelete (RemoteName);
    if not cacheAdd (RemoteName) then exit;
  end;  

  Result := true;  
end;  

{ ------------------------------------------------------------------ }

function FsSetAttr (RemoteName : PChar; NewAttr : integer) : bool; stdcall;

begin
  inc (RemoteName);                         // skip over leading backslash
  Result := false;
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('Set FileAttr ' + RemoteName));

  HPLXSetAttr (RemoteName, Lo (NewAttr));
  if HPLXError then begin
    filerReset;
    exit;
  end;
  
  if isInCache (extractFilePath (RemoteName)) then begin
    cacheDelete (RemoteName);
    if not cacheAdd (RemoteName) then exit;
  end;  

  Result := true;
end;  

{ ------------------------------------------------------------------ }

function FsGetFile (RemoteName, LocalName : PChar; CopyFlags : integer ;
                    RemoteInfo : pRemoteInfo) : integer; stdcall;

var
  FileLength, Transferred : longword;
  Delivered    : word;
  PercentDone  : integer;
  WorkFile     : file;
  Buffer       : TData;

begin
  if (CopyFlags and FS_COPYFLAGS_OVERWRITE) = 0 then
    if fileExists (LocalName) then begin
      Result := FS_FILE_EXISTS;
      exit
    end;  

  inc (RemoteName);                         // skip over leading backslash
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('Copy ' + RemoteName + ' to ' + LocalName));

  assignFile (WorkFile, string (LocalName));
  rewrite (WorkFile, 1);
  if IOResult <> 0 then begin
    Result := FS_FILE_WRITEERROR;
    exit;
  end;

  HPLXReset (RemoteName);
  if HPLXError then begin
    filerReset;
    Result := FS_FILE_NOTFOUND;
    exit;
  end;
 
 if ProgressProc (PluginNumber, RemoteName, LocalName, 0) <> 0 then begin
    Result := FS_FILE_USERABORT;
    closeFile (WorkFile);
    deleteFile (LocalName);
    HPLXCloseFile;
    exit;
  end;

  Transferred := 0;
  FileLength  := RemoteInfo^.SizeLow;

// ------- main copy loop -------
  repeat
    HPLXBlockRead (Buffer, PACKET_DATA_SIZE, Delivered); 
    if HPLXError then begin
      filerReset;
      closeFile (WorkFile);
      deleteFile (LocalName);
      Result := FS_FILE_READERROR;
      exit;
    end;

    Transferred := Transferred + Delivered;
    if FileLength = 0 then
      PercentDone := 100
    else
      PercentDone := (Transferred * 100) div FileLength;

    blockWrite (WorkFile, Buffer, Delivered);
    if IOResult <> 0 then begin
      closeFile (WorkFile);
      deleteFile (LocalName);
      HPLXCloseFile;
      Result := FS_FILE_WRITEERROR;
      exit;
    end;

    if ProgressProc (PluginNumber, RemoteName, LocalName, PercentDone) <> 0 then begin
      closeFile (WorkFile);
      deleteFile (LocalName);
      HPLXCloseFile;
      Result := FS_FILE_USERABORT;
      exit;
    end;

  until Delivered < PACKET_DATA_SIZE;
// ------- end of main copy loop -------    

  setFileTime (TFileRec (WorkFile).Handle, nil, nil, @RemoteInfo^.LastWriteTime);

  closeFile (WorkFile);
  HPLXCloseFile;
  if HPLXError then begin
    filerReset;
    Result := FS_FILE_READERROR;
    exit;
  end;
  if IOResult <> 0 then begin
    filerReset;
    Result := FS_FILE_WRITEERROR;
    exit;
  end;

  setFileAttributes (LocalName, RemoteInfo^.Attr);

  if (CopyFlags and FS_COPYFLAGS_MOVE) <> 0 then begin          // delete remote file
    dec (RemoteName);
    if not FsDeleteFile (RemoteName) then begin
      Result := FS_FILE_READERROR;
      exit;
    end;  
  end;

  Result := FS_FILE_OK;

end;                                        // function FsGetFile

{ ------------------------------------------------------------------ }

function FsPutFile (LocalName, RemoteName : PChar; CopyFlags : integer) : integer; stdcall;

var
  FileLength, Transferred, Delivered : longword;
  PercentDone : integer;
  WorkFile    : file;
  Buffer      : TData;
  SRemoteName : string;

begin
  SRemoteName := RemoteName + 1;
  SRemoteName := extractFilePath (SRemoteName) + getShortFileName (extractFileName (SRemoteName));
  Result := FS_FILE_WRITEERROR;

  if (CopyFlags and FS_COPYFLAGS_OVERWRITE) = 0 then begin      // does remote file exist?
    if HPLXFileExists (SRemoteName) then begin
      Result := FS_FILE_EXISTS;
      exit
    end;  
    if HPLXError then begin
      filerReset;
      exit;
    end;  
  end
  else                                      // if exists, delete it from cache
    cacheDelete (SRemoteName);

  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('Copy ' + LocalName + ' to ' + SRemoteName));

  assignFile (WorkFile, string (LocalName));
  FileMode := fmOpenRead;
  reset (WorkFile, 1);
  if IOResult <> 0 then begin
    Result := FS_FILE_READERROR;
    exit;
  end;

  HPLXRewrite (SRemoteName);
  if HPLXError then begin
    filerReset;
    Result := FS_FILE_NOTFOUND;
    exit;
  end;

  if ProgressProc (PluginNumber, PChar (SRemoteName), LocalName, 0) <> 0 then begin
    closeFile (WorkFile);
    HPLXCLoseFile;
    HPLXDeleteFile (SRemoteName);
    if HPLXError then
      filerReset;
    Result := FS_FILE_USERABORT;
    exit;
  end;

  Transferred := 0;
  FileLength  := fileSize (LocalName); // from KOL
//  FileLength  := System.fileSize (WorkFile);
  
// --------- main copy loop -----------  
  repeat
    blockRead (WorkFile, Buffer, PACKET_DATA_SIZE, Delivered);
    if IOResult <> 0 then begin
      closeFile (WorkFile);
      HPLXCloseFile;
      HPLXDeleteFile (SRemoteName);
      if HPLXError then
        filerReset;
      Result := FS_FILE_READERROR;
      exit;
    end;

    HPLXBlockWrite (Buffer, Delivered);
    if HPLXError then begin
      filerReset;
      closeFile (WorkFile);
      HPLXDeleteFile (SRemoteName);
      exit;
    end;

    Transferred := Transferred + Delivered;
    if FileLength = 0 then
      PercentDone := 100
    else
      PercentDone := (Transferred * 100) div FileLength;

    if ProgressProc (PluginNumber, PChar(SRemoteName), LocalName, PercentDone) <> 0 then begin
      closeFile (WorkFile);
      HPLXCloseFile;
      HPLXDeleteFile (SRemoteName);
      if HPLXError then
        filerReset;
      Result := FS_FILE_USERABORT;
      exit;
    end;

  until Delivered < PACKET_DATA_SIZE;
// -------- end of main copy loop --------

  HPLXSetFileTime (fileGetDate (TFileRec (WorkFile).Handle));

  closeFile (WorkFile);
  HPLXCloseFile;
  if HPLXError then begin
    filerReset;
    exit;
  end;
  if IOResult <> 0 then begin
    Result := FS_FILE_READERROR;
    exit;
  end;
  
  HPLXSetAttr (SRemoteName, Lo (getFileAttributes (LocalName)) and $27); // copy file attributes
  if HPLXError then begin
    filerReset;
    exit;
  end;

  if (CopyFlags and FS_COPYFLAGS_MOVE) <> 0 then
    if not deleteFile (LocalName) then begin
      Result := FS_FILE_READERROR;
      exit;
    end;

  if isInCache (extractFilePath (SRemoteName)) then
    if not cacheAdd (SRemoteName) then exit;

  Result := FS_FILE_OK;
end;                                        // function FsPutFile

{ ------------------------------------------------------------------ }

function FsRenMovFile (OldName, NewName : PChar; Move, OverWrite : bool;
         RemoteInfo : pRemoteInfo) : integer; stdcall;

begin
  inc (OldName);                            // skip over leading backslash
  inc (NewName);

  if OldName[1] <> NewName[1] then begin    // moving to different drive is not supported
    Result := FS_FILE_NOTSUPPORTED;
    exit;
  end;  
  
  Result := FS_FILE_OK;
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('Ren ' + OldName + ' to ' + NewName));
  HPLXRenameFile (OldName, NewName);
  if HPLXError then begin
    filerReset;
    Result := FS_FILE_NOTFOUND;
    exit;
  end;

  cacheDelete (OldName);
  if isInCache (extractFilePath (NewName)) then 
    if not cacheAdd (NewName) then begin
      Result := FS_FILE_WRITEERROR;
      exit;
    end;
end;
  
{ ------------------------------------------------------------------ }

function FsExecuteFile (MainWin : thandle; RemoteName, Verb : PChar) : integer; stdcall;

begin
  Result := FS_EXEC_OK;

  if Verb = 'open' then
    Result := FS_EXEC_YOURSELF;
  
  if (Verb = 'properties') and (RemoteName = '\') then
    LaunchSetForm (MainWin, IniFileName);
// else do nothing    
end;

{ ------------------------------------------------------------------ }

procedure FsGetDefRootName (DefRootName : PChar; maxlen : integer); stdcall;

begin
  strLCopy (DefRootName, PLUGIN_TITLE, maxlen - 1);
end;

{ ------------------------------------------------------------------ }

procedure FsSetDefaultParams (dps : pFsDefaultParamStruct); stdcall;

begin
  IniFileName := dps^.DefaultIniName;
end;

{ ------------------------------------------------------------------ }
{$Hints Off}
function FsExtractCustomIcon (RemoteName : PChar; ExtractFlags : integer; var TheIcon : hicon) : integer; stdcall;

var
  SRemoteName, Ext : string;

begin
  Result := FS_ICON_USEDEFAULT;
  
  SRemoteName := string (RemoteName + 1);
  Ext := copy (upperCase (ExtractFileExt (SRemoteName)), 2, 3);
  
  if (length (SRemoteName) = 3) and (SRemoteName[2] = ':') then begin
    TheIcon    := LoadImgIcon ('DISK', 16);
    RemoteName := 'HPLXDISK';
    Result     := FS_ICON_EXTRACTED;
  end;

  if Ext <> '' then begin
    TheIcon    := LoadImgIcon (PChar (Ext), 16);
    if TheIcon <> 0 then begin
      RemoteName := PChar ('HPLX' + Ext);
      Result     := FS_ICON_EXTRACTED;
    end;  
  end;
end;

{$Hints On}
{ ------------------------------------------------------------------ }

exports

  FsDeleteFile,
  FsDisconnect,
  FsExecuteFile,
  FsExtractCustomIcon,
  FsFindClose,
  FsFindFirst,
  FsFindNext,
  FsGetDefRootName,
  FsGetFile,
  FsInit,
  FsMkDir,
  FsPutFile,
  FsRemoveDir,
  FsRenMovFile,
  FsSetAttr,
  FsSetDefaultParams,
  FsSetTime;
  
{ ------------------------------------------------------------------ }

end.