/* LBCD.C -- BCD Math

	Written May 1994 by Craig A. Finseth
	Copyright 1994 by Craig A. Finseth
*/

#include "loki.h"

#if defined(MSDOS)
#define MAXINT		2147483647L
#endif

#define COMMA		','
#define DECIMAL		'.'
#define EXPONENT1	'e'
#define EXPONENT2	'E'
#define MINUS		'-'
#define PLUS		'+'
#define INFINITYSTR	"oo"
#define AFTERDECIMAL	5	/* affects output printing */
#define INFINITY	(0x7fff)	/* exponent of big number */

static struct number result;	/* holds result of all operations */

void B_AddU(struct number *larger, struct number *smaller);
FLAG B_AdjustExponent(struct number *loc, int e1, int e2);
void B_CopySign(struct number *from, struct number *to, int sgn);
FLAG B_GEQ(struct number *a, struct number *b);
void B_LeftNormalize(struct number *loc, int extra);
void B_Reverse(char *to, char *from);
void B_RightNormalize(struct number *loc, int overflow);
void B_Round(struct number *loc);
void B_ShiftLeft(struct number *loc);
void B_ShiftRight(struct number *loc);
void B_SubU(struct number *larger, struct number *smaller);

/* ------------------------------------------------------------ */

/* Add B to A.  Modify A. */

void
BAdd(struct number *a, struct number *b)
	{
	struct number *larger;
	struct number *smaller;
	int type = a->type;

	if (a->exp == INFINITY || b->digits[0] == 0) return;

	if (a->digits[0] == 0 || b->exp == INFINITY) {
		*a = *b;
		return;
		}
	if (a->exp > b->exp || a->exp == b->exp && B_GEQ(a, b)) {
		larger = a;
		smaller = b;
		}
	else	{
		larger = b;
		smaller = a;
		}
	if (larger->sign != smaller->sign)
		B_SubU(larger, smaller);
	else	B_AddU(larger, smaller);

	
	B_CopySign(&result, a, larger->sign && (result.digits[0] != 0));
	a->type = type;
	}


/* ------------------------------------------------------------ */

/* Assign N to the number.  N is <= 9. */

void
BAssign(struct number *x, int n)
	{
	int cnt;

	for (cnt = 1; cnt < BCDDIGITS; ) {
		x->digits[cnt++] = 0;
		}
	x->digits[0] = n;
	x->sign = 0;
	x->exp = 0;
	x->type = RES_UN_NONE / UNIT_SIZE;
	}


/* ------------------------------------------------------------ */

/* Assign N to the number.  N is any positive value. */

void
BAssignI(struct number *x, int n)
	{
	char digbuf[SMALLBUFFSIZE];
	int cnt;
	int cnt2;

	for (cnt = 1; cnt < BCDDIGITS; ) {
		x->digits[cnt++] = 0;
		}
	x->sign = 0;
	x->exp = 0;

	if (n < 0) {
		x->sign = 1;
		n = -n;
		}

	for (cnt = 0; cnt < BCDDIGITS && n > 9; cnt++) {
		digbuf[cnt] = n % 10;
		n /= 10;
		}

	for (cnt2 = 0; cnt2 < cnt; cnt2++) {
		x->digits[cnt - cnt2] = digbuf[cnt2];
		}
	x->digits[0] = n;
	x->exp = cnt;
	x->type = RES_UN_NONE / UNIT_SIZE;
	}


/* ------------------------------------------------------------ */

/* Assign N to the number.  N is any positive value. */

void
BAssignL(struct number *x, long n)
	{
	char digbuf[SMALLBUFFSIZE];
	int cnt;
	int cnt2;

	for (cnt = 1; cnt < BCDDIGITS; ) {
		x->digits[cnt++] = 0;
		}
	x->sign = 0;
	x->exp = 0;

	if (n < 0) {
		x->sign = 1;
		n = -n;
		}

	for (cnt = 0; cnt < BCDDIGITS && n > 9; cnt++) {
		digbuf[cnt] = n % 10;
		n /= 10;
		}

	for (cnt2 = 0; cnt2 < cnt; cnt2++) {
		x->digits[cnt - cnt2] = digbuf[cnt2];
		}
	x->digits[0] = n;
	x->exp = cnt;
	x->type = RES_UN_NONE / UNIT_SIZE;
	}


/* ------------------------------------------------------------ */

/* Divide A by B.  Modify A. */

void
BDiv(struct number *a, struct number *b)
	{
	int i;
	int j;
	int carry;
	int x;
	int n;
	int type = a->type;

	if (b->exp == INFINITY) {
		BAssign(a, 0);
		return;
		}
	if (b->digits[0] == 0 || a->exp == INFINITY) {
		result.exp = INFINITY;
		}
	else	{
		for (i = 0, x = 0; i < BCDDIGITS; i++) {
			result.digits[i] = 0;
			while (x != 0 || B_GEQ(a, b)) {
				n = (x * 10 + a->digits[0]) /
					(b->digits[0] + 1);
				if (n == 0) n = 1;

				carry = 0;
				/* a -= n * b */
				for (j = BCDDIGITS - 1; j >= 0; j--) {
					carry += a->digits[j] -
						n * b->digits[j];
					a->digits[j] = (carry + 100) % 10;
					carry = (carry - 9) / 10;
					}
				x += carry;
				result.digits[i] += n;
				}
			x = a->digits[0];
			B_ShiftLeft(a);
			}
		if (B_AdjustExponent(&result, a->exp, -b->exp)) {
			B_LeftNormalize(&result, (x * 100 +
				a->digits[0] * 10) /
				(b->digits[0] * 10 + b->digits[1]));
			}
		}
	B_CopySign(&result, a, a->sign ^ b->sign);
	a->type = type;
	}


/* ------------------------------------------------------------ */

/* Converts the string to a number.  Return True on success or False
on failure. */

FLAG
BEnter(struct number *n, char *str)
	{
	char c;
	FLAG afterdecimal = FALSE;
	FLAG innumber = FALSE;
	FLAG inexponent = FALSE;
	FLAG expsign = FALSE;
	int i;
	int pos;
	int manexp = -1;

	BAssign(n, 0);
	pos = 0;

	while (*str != NUL) {
		c = *str++;
		if (inexponent) {
			if (c >= '0' && c <= '9') {
				n->exp = n->exp * 10 + c - '0';
				n->exp %= 1000;
				}
			else if (c == MINUS) {
				expsign = TRUE;
				}
			else if (c == PLUS) {
				expsign = FALSE;
				}
			else	{
				return(FALSE);
				}
			}
		else if (c == '0' && pos == 0) {
			if (afterdecimal) manexp--;
			}
		else if (c >= '0' && c <= '9') {
			innumber = TRUE;
			if (pos < BCDDIGITS) n->digits[pos++] = c - '0';
			if (!afterdecimal) manexp++;
			}
		else if (c == EXPONENT1 || c == EXPONENT2) {
			if (!innumber) {
				n->digits[0] = 1;
				pos = 1;
				}
			inexponent = TRUE;
			}
		else if (c == MINUS) {
			n->sign = 1;
			}
		else if (c == DECIMAL) {
			afterdecimal = TRUE;
			}
		else	{
			return(FALSE);
			}
		}
	if (expsign) n->exp = -n->exp;
	n->exp += manexp;
	if (pos == 0) n->exp = 0;	/* normalize zero */
	n->type = RES_UN_NONE / UNIT_SIZE;
	return(TRUE);
	}


/* ------------------------------------------------------------ */

/* Return True if the numbers are equal. */

FLAG
BEqual(struct number *a, struct number *b)
	{
	int cnt;

	if (a->sign != b->sign) return(FALSE);
	if (a->exp != b->exp) return(FALSE);
	for (cnt = 0; cnt < BCDDIGITS; cnt++) {
		if (a->digits[cnt] != b->digits[cnt]) return(FALSE);
		}
	return(TRUE);
	}


/* ------------------------------------------------------------ */

/* Format a number into a supplied buffer.  Rounds the last two
digits. */

void
BFormat(char *buf, struct number *n)
	{
	int i;
	int e;
	int c;
	char numbuf[SMALLBUFFSIZE];
	char *cptr = numbuf;
	FLAG innumber = FALSE;
	FLAG sgn;

	if (n->exp == INFINITY) {
		xstrcpy(buf, INFINITYSTR);
		return;
		}
	result = *n;
	if (result.digits[BCDDIGITS - 1] != 0) B_Round(&result);
	if (result.digits[BCDDIGITS - 2] != 0) B_Round(&result);
	e = result.exp;
	if (e > BCDDIGITS - 2 || e < -AFTERDECIMAL) {
		sgn = e < 0;
		if (sgn) e = -e;
		while (e > 0) {
			*cptr++ = e % 10 + '0';
			e /= 10;
			}
		if (sgn) *cptr++ = MINUS;
		*cptr++ = EXPONENT1;
		e = 0;
		}
	for (i = BCDDIGITS - 2; i >= 0; i--) {
		if (i == e) {
			if (innumber) *cptr++ = m.radix ? COMMA : DECIMAL;
			innumber = TRUE;
			}
		if (innumber || result.digits[i] != 0 || i == 0) {
			*cptr++ = result.digits[i] + '0';
			innumber = TRUE;
			}
		}
	for ( ; i > e; i--) *cptr++ = '0';
	if (i == e) *cptr++ = m.radix ? COMMA : DECIMAL;
	if (n->sign) *cptr++ = MINUS;

	*cptr = NUL;
	B_Reverse(buf, numbuf);
	}


/* ------------------------------------------------------------ */

/* Return True if the number is negative. */

FLAG
BIsNeg(struct number *a)
	{
	return(a->sign != 0);
	}


/* ------------------------------------------------------------ */

/* Return True if the number is zero. */

FLAG
BIsZero(struct number *a)
	{
	if (a->exp != 0) return(FALSE);
	return(a->digits[0] == 0);
	}


/* ------------------------------------------------------------ */

/* Return the value as a long. */

long
BLong(struct number *a)
	{
	int cnt;
	long result = 0;

	if (a->exp == INFINITY || a->exp > BCDDIGITS) {
		return(MAXINT);
		}
	if (a->exp < 0) {
		return(0);
		}

	for (cnt = 0; cnt < BCDDIGITS && cnt < a->exp + 1; cnt++) {
		result = result * 10 + a->digits[cnt];
		}
	return(a->sign ? -result : result);
	}


/* ------------------------------------------------------------ */

/* Return the fractional part of N in F and drop it from N. */

void
BModf(struct number *n, struct number *f)
	{
	int cnt;

	BAssign(f, 0);
	if (n->exp == INFINITY || n->exp > BCDDIGITS) {
		return;
		}
	if (n->exp < 0) {
		*f = *n;
		BAssign(n, 0);
		return;
		}

	f->sign = n->sign;
	f->exp = -1;
	for (cnt = n->exp + 1; cnt < BCDDIGITS; cnt++) {
		f->digits[cnt - n->exp - 1] = n->digits[cnt];
		n->digits[cnt] = 0;
		}
	B_LeftNormalize(f, 0);
	}


/* ------------------------------------------------------------ */

/* Multiply A by B.  Modify A. */

void
BMul(struct number *a, struct number *b)
	{
	int i;
	int j;
	int carry;
	int type = a->type;

	if (a->exp == INFINITY || b->exp == INFINITY) {
		a->exp = INFINITY;
		return;
		}
	BAssign(&result, 0);

	if (a->digits[0] != 0 && b->digits[0] != 0) {
		for (i = BCDDIGITS - 1; i >= 0; i--) {
			carry = 0;
			for (j = BCDDIGITS - 1; j >= 0; j--) {
				carry += a->digits[i] * b->digits[j] +
					result.digits[j];
				result.digits[j] = carry % 10;
				carry /= 10;
				}
			if (i > 0) {
				B_ShiftRight(&result);
				result.digits[0] = carry;
				}
			}
		if (B_AdjustExponent(&result, a->exp, b->exp)) {
			B_RightNormalize(&result, carry);
			}
		}
	B_CopySign(&result, a, a->sign ^ b->sign);
	a->type = type;
	}


/* ------------------------------------------------------------ */

/* Negate A. */

void
BNeg(struct number *a)
	{
	a->sign = !a->sign;
	}


/* ------------------------------------------------------------ */

/* Subtract B from A.  Modify A. */

void
BSub(struct number *a, struct number *b)
	{
	struct number bn = *b;

	bn.sign = !bn.sign;
	BAdd(a, &bn);
	}


/* ------------------------------------------------------------ */

/* Unsigned addition.  The second is destroyed. */

void
B_AddU(struct number *larger, struct number *smaller)
	{
	int i;
	int x;
	int carry;

	x = larger->exp - smaller->exp;
	carry = 0;
	for (i = BCDDIGITS - 1; i >= 0; i--) {
		carry += larger->digits[i] +
			((i >= x) ? smaller->digits[i - x] : 0);
		result.digits[i] = carry % 10;
		carry /= 10;
		}
	result.exp = larger->exp;
	B_RightNormalize(&result, carry);
	}


/* ------------------------------------------------------------ */

/* Sums exponents, handles over/under flow. */

FLAG
B_AdjustExponent(struct number *loc, int e1, int e2)
	{
	loc->exp = e1 + e2;
	if (e1 == INFINITY || e2 == INFINITY ||
		 e1 > 0 && e2 > 0 && loc->exp < 0) {
		loc->exp = INFINITY;
		return(FALSE);
		}
	else if (e1 < 0 && e2 < 0 &&
		 (loc->exp == INFINITY || loc->exp >= 0)) {
		BAssign(loc, 0);
		return(FALSE);
		}
	else	return(TRUE);
	}


/* ------------------------------------------------------------ */

/* Copy FROM to TO, changing the sign to SGN. */

void
B_CopySign(struct number *from, struct number *to, int sgn)
	{
	*to = *from;
	to->sign = sgn;
	}


/* ------------------------------------------------------------ */

/* Compare two numbers. */

FLAG
B_GEQ(struct number *a, struct number *b)
	{
	int cnt;

	for (cnt = 0; cnt < BCDDIGITS; cnt++) {
		if (a->digits[cnt] != b->digits[cnt]) {
			return (a->digits[cnt] >= b->digits[cnt]);
			}
		}
	return(TRUE);
	}


/* ------------------------------------------------------------ */

/* Normalizes the number leftward.  EXTRA is the extra digit to put on
the right. */

void
B_LeftNormalize(struct number *loc, int extra)
	{
	int i;
	int j;

	if (loc->exp == INFINITY) return;

	for (i = 0; i < BCDDIGITS && loc->digits[i] == 0; i++) ;
	if (i >= BCDDIGITS) {	/* is it zero? */
		loc->exp = 0;
		return;
		}
	if (i == 0) return;
	for (j = 0; j < BCDDIGITS - i; j++) {
		loc->digits[j] = loc->digits[j + i];
		}
	loc->digits[j++] = extra;

	for (; j < BCDDIGITS; j++) {
		loc->digits[j] = 0;
		}
	B_AdjustExponent(loc, loc->exp, -i);
	}


/* ------------------------------------------------------------ */

/* Reverse the characters in the FROM string and place the resulting
string in TO.  TO must be at least as large as strlen(FROM). */

void
B_Reverse(char *to, char *from)
	{
	int len = strlen(from);
	int cnt;

	for (cnt = 0; cnt < len; cnt++) {
		to[cnt] = from[len - cnt - 1];
		}
	to[len] = NUL;
	}


/* ------------------------------------------------------------ */

/* Normalizes any overflow. */

void
B_RightNormalize(struct number *loc, int overflow)
	{
	if (loc->exp == INFINITY) return;

	if (overflow != 0) {
		B_ShiftRight(loc);
		loc->digits[0] = overflow;
		B_AdjustExponent(loc, loc->exp, 1);
		}
	}


/* ------------------------------------------------------------ */

/* Round the number. */

void
B_Round(struct number *loc)
	{
	int i;
	int carry;

	for (i = BCDDIGITS - 1; i >= 0 && loc->digits[i] == 0; i--) ;
	if (i <= 0) return;

	carry = (loc->digits[i] + 5) / 10;
	loc->digits[i] = 0;
	for (i--; i >= 0; i--) {
		carry += loc->digits[i];
		loc->digits[i] = carry % 10;
		carry /= 10;
		}
	B_RightNormalize(loc, carry);
	}


/* ------------------------------------------------------------ */

/* Shifts the digits left (multiplies by 10). */

void
B_ShiftLeft(struct number *loc)
	{
	int i;

	for (i = 0; i < BCDDIGITS - 1; i++) {
		loc->digits[i] = loc->digits[i + 1];
		}
	loc->digits[BCDDIGITS - 1] = 0;
	}


/* ------------------------------------------------------------ */

/* Shifts the digits right (divides by 10). */

void
B_ShiftRight(struct number *loc)
	{
	int i;

	for (i = BCDDIGITS - 1; i > 0; i--) {
		loc->digits[i] = loc->digits[i - 1];
		}
	loc->digits[0] = 0;
	}


/* ------------------------------------------------------------ */

/* Unsigned subtraction. Won't zero-cross. */

void
B_SubU(struct number *larger, struct number *smaller)
	{
	int i;
	int x = larger->exp - smaller->exp;
	int carry = 0;

	for (i = BCDDIGITS - 1; i >= 0; i--) {
		carry += larger->digits[i] -
			((i >= x) ? smaller->digits[i - x] : 0);
		result.digits[i] = (carry+10) % 10;
		carry = (carry <0) ? -1 : 0;
		}
	result.exp = larger->exp;
	B_LeftNormalize(&result, 0);
	}

/* end of LBCD.C -- BCD Math */
