/* --------------------------------- views.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Show all visual displays
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "fly.h"
#include "hud.h"

static MENU MenuView[] = {
	{'f', "front"},
	{'n', "none"},
	{'a', "rear"},
	{'m', "map"},
	{'r', "radar"},
	{'t', "target"},
	{'p', "pan"},
	{'g', "gaze"},
	{'c', "chase"},
	{'o', "follow"},
	{'h', "hud"},
	{'u', "up-front"},
	{'s', "misc"},
	{'R', "right"},
	{'L', "left"},
{'\0', 0}};

static void FAR
ident (int id, int orgx, int orgy, int maxx, int maxy, int ss)
{
	int	bss, i, l;
	char	*p;

	if (st.flags1 & SF_IDENT) {
		i = id ? st.hdd[id-1].type : st.extview;
		p = MenuView[i].text;
		l = strlen (p);
		bss = ss*4;
		if (bss > 64)
			bss = 64;
		if (bss > maxx/l)
			bss = maxx/l;
		stroke_num (orgx-bss*l/2, orgy, (long)id, bss, st.white);
		stroke_str (orgx-bss*l/2, orgy+bss, p, bss, st.white);
	} else if (!(st.flags & SF_BLANKER)) {
		i = id ? st.hdd[id-1].type
		       : (st.flags1&SF_EXTVIEW ? st.extview: -1);
		if (i >= 0) {
			p = MenuView[i].text;
			l = stroke_size (p, ss);
			stroke_str (orgx+maxx-l, orgy-maxy+ss, p, ss, st.hfg);
		}
	}
}

/*
 * this is the main display dispatcher.
*/

extern void FAR
show_main (VIEW *view, OBJECT *pov, int shift, int type)
{
	int	orgx, orgy, maxx, maxy, ss;
	OBJECT	*p;

	if (ET_PLANE == CV->e_type)
		p = CV;
	else
		p = CC;
	if (!p)
		return;

	orgx  = fmul (view->window->orgx, view->screen->sizex)
			+ view->screen->minx;
	orgy  = fmul (view->window->orgy, view->screen->sizey)
			+ view->screen->miny;

	maxx = fmul (view->window->maxx - iabs(shift), view->screen->sizex) - 1;
	maxy = fmul (view->window->maxy, view->screen->sizey) - 1;

	ss = fmul (maxx, FCON (0.04));
	ss = muldiv (ss, st.StFontSize, 8);
	ss = (ss+1)&~1;		/* make it even */
	if (ss < 8)
		ss = 8;

	show_hud (view, pov, p, shift, orgx, orgy, maxx, maxy, type);
/*
 * misc data
*/
	if ((HDD_FRONT == type && st.flags & SF_MAIN && EE(p)->hud1 & HUD_MISC)
	    || HDD_MISC == type)
		show_misc (view, p, maxx, maxy, orgx, orgy, ss);
/*
 * status info. This text is shown over the whole screen. It will now show if
 * there is an active HDD_INFO.
*/
	screen_info (view, orgx, orgy, maxx, maxy, ss, type);

	if (st.flags & SF_MAIN)
		ident (0, orgx, orgy, maxx, maxy, ss);
}

extern int FAR
scenery (int type)
{
	return (HDD_NONE != type && HDD_HUD != type && HDD_INFO != type &&
		HDD_MISC != type);
}

static void FAR
show_view (VIEW *cv, int type)
{
	OBJECT	*pov;
	int	t, vr, wr, smaxx, smaxy, shud, shud1;

	if (!cv)
		return;

	t = (HDD_HUD == type) ? HDD_FRONT : type;
	if (!(pov = get_viewer (t)) || !pov->viewport)
		return;

	cv->viewport = pov->viewport;
	smaxx = cv->viewport->maxx;
	smaxy = cv->viewport->maxy;
	if (ET_PLANE == CV->e_type) {
		shud  = EE(CV)->hud;
		shud1 = EE(CV)->hud1;
		EE(CV)->hud  &= ~(HUD_ON|HUD_CURSOR|HUD_RETICLE);
		EE(CV)->hud1 &= ~(HUD_VALARM|HUD_MISC);
	} else
		shud = shud1 = 0;	/* avoid compiler warning */
/*
 * Adjust aspect ratio. All viewports use the same aspect ratio as the main
 * window, now adjust the viewport for the actual window.
 * The 1/4 scaling is done to allow handling high aspect ratios.
*/
	if (!(st.flags & SF_MAIN)) {
		vr = muldiv (FONE/4, cv->viewport->maxx, cv->viewport->maxy);
		wr = muldiv (FONE/4, cv->window->maxx,   cv->window->maxy);
		wr = muldiv (wr, cv->screen->device->lengx,
						cv->screen->device->lengy);
#if 0
		if (vr > wr)	/* viewport too wide */
			cv->viewport->maxx = muldiv (cv->viewport->maxx,
							wr, vr);
		else		/* viewport too hight */
			cv->viewport->maxy = muldiv (cv->viewport->maxy,
							vr, wr);
#else
		if (vr > wr)	/* viewport wider than window */
			cv->viewport->maxy = muldiv (cv->viewport->maxy,
							vr, wr);
		else		/* viewport higher than window */
			cv->viewport->maxy = muldiv (cv->viewport->maxy,
							vr, wr);
#endif
	}

	if (scenery (type)) {
		if (st.flags & SF_SKY)
			show_sky (cv, pov, st.lblue, 0);
		objects_show (cv, pov, 0, 0, 0);
	}

	show_main (cv, pov, 0, type);

	cv->viewport->maxx = smaxx;
	cv->viewport->maxy = smaxy;
	if (ET_PLANE == CV->e_type) {
		EE(CV)->hud  = shud;
		EE(CV)->hud1 = shud1;
	}

	cv->viewport = 0;
}

extern void FAR
show_hdd (void)
{
	int	i, orgx, orgy, maxx, maxy, ss;
	long	t, t27;
	VIEW	*v;

	t27 = st.stats[27];
	t = Tm->Milli ();
	for (i = 0; i < NHDD; ++i) {
		if (!(st.hdd[i].flags & HDD_ON))
			continue;

		v = &st.hdd[i].view;
		orgx = fmul (v->window->orgx, v->screen->sizex)
			+ v->screen->minx;
		orgy = fmul (v->window->orgy, v->screen->sizey)
			+ v->screen->miny;
		maxx = fmul (v->window->maxx, v->screen->sizex) - 1;
		maxy = fmul (v->window->maxy, v->screen->sizey) - 1;
		ss = fmul (maxx, FCON (0.04));
		ss = muldiv (ss, st.StFontSize, 8);
		ss = (ss+1)&~1;		/* make it even */
		if (ss < 8)
			ss = 8;

		switch (st.hdd[i].type) {
		case HDD_NONE:
			break;
		case HDD_INFO:
			if (hdd_menu (v, orgx, orgy, maxx, maxy))
				screen_info (v, orgx, orgy, maxx, maxy, ss,
					st.hdd[i].type);
			break;
		case HDD_MISC:
			show_misc (v, CC, maxx, maxy, orgx, orgy, ss);
			break;
		case HDD_HUD:
		default:
			show_view (v, st.hdd[i].type);
			break;
		}
		ident (i+1, orgx, orgy, maxx, maxy, ss);
		st.hdd[i].bufs[st.which_buffer] = st.buf[HEAD];
		st.buf[HEAD] = st.buf[TAIL] = 0;
	}
	st.stats[40] += (Tm->Milli () - t) - (st.stats[27] - t27);
}

/*
 * Select external view mode
*/

extern int FAR
menu_view (int sel)
{
	sel = menu_open (MenuView, sel);
	if (MENU_FAILED != sel)
		menu_close ();
	return (sel);
}
