/* --------------------------------- grx.c --------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Graphics driver for X11.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include "fly.h"
#include "colors.h"

static DEVICE *CurrentDevice = 0;

#define APPNAME "fly8"
#define NOCOLOR	0xffff

static Display *TheDisplay;
static GC TheGC;
static Window TheWindow, TheRoot;
static int TheScreen;
static Colormap TheColormap;
static unsigned short colors[256];
static unsigned long FgColor, BgColor;
static int xold = 0, yold = 0;
static int last_color = -1;


static void
MoveTo (int x, int y)
{
  xold = x;
  yold = y;
}

static void
DrawTo (int x, int y, Uint c)
{
	if (last_color != c) {
		last_color = c;
		XSetForeground(TheDisplay, TheGC, colors[c]);
	}
	XDrawLine(TheDisplay, TheWindow, TheGC, xold, yold, x, y);
	xold = x;
	yold = y;
}

static void FAR
SetActive (int page)
{
}

static void FAR
SetVisual (int page)		/* done */
{
}

static int FAR
CursorMode (int mode)
{
	return (0);
}

static void FAR
TextColor (Uint fg, Uint bg)			/* done */
{
}

static void FAR
FontSet (DEVICE *dev, char *fname)		/* done */
{
	dev->FontWidth  = 8;
	dev->FontHeight = 8;
}

static void FAR
TextChar (int c)				/* done */
{
  putc (c, stderr);
  fflush (stderr);
}

static void FAR
TextPut (int c)					/* done */
{
  putc (c, stderr);
  fflush (stderr);
}

static void FAR
SetTextPos (int row, int col)			/* done */
{
}

static void FAR
PushTextPos (void)
{
}

static void FAR
PopTextPos (void)
{
}

static void FAR
TextClear (void)
{
}

static void FAR
WriteMode (int mode)
{
	switch (mode) {
	default:
	case T_MSET:
		mode = GXcopy;
		break;
	case T_MOR:
		mode = GXor;
		break;
	case T_MXOR:
		mode = GXxor;
		break;
	}
	XSetFunction (TheDisplay, TheGC, mode);
}

static void FAR
SetPalette (int n, long c)
{
  XColor color;

  if (CurrentDevice->colors < 16)
    return;
  if (NOCOLOR == colors[n])
    /* FreeThisColor */;
  color.red   = 0xff00 & (int)(c <<  8); 
  color.green = 0xff00 & (int)(c      ); 
  color.blue  = 0xff00 & (int)(c >>  8); 
  XAllocColor (TheDisplay, TheColormap, &color);
  colors[n] = color.pixel;
}

static void FAR
OpenTextWindow (SCREEN *scr)
{
}

static void FAR
CloseTextWindow (SCREEN *scr)
{
}

static void FAR
Ellipse (int x, int y, int rx, int ry, Uint c)
{
	if (last_color != c) {
		last_color = c;
		XSetForeground (TheDisplay, TheGC, colors[c]);
	}
	XDrawArc (TheDisplay, TheWindow, TheGC, x-rx, y-ry, 2*rx, 2*ry,
			0, 360*64);
}

static void FAR
Sync (void)
{
}

static void
ClearWin (void)
{
  XSetWindowBackground (TheDisplay, TheWindow, colors[st.black]);
  XClearWindow (TheDisplay, TheWindow);
  st.flags |= SF_CLEARED;
}

static int FAR
Init (DEVICE *dev)
{
  XEvent	event;
  int		i;

  if (dev->sizex == 0 || dev->sizey == 0)
    return (1);

  TheDisplay = XOpenDisplay (""); 
  if (!TheDisplay) {
	fprintf (stderr, "%s: can't open display\n", APPNAME);
	exit (0);
  }
  CurrentDevice = dev;

  TheScreen = DefaultScreen (TheDisplay);
  TheRoot = DefaultRootWindow (TheDisplay);
  FgColor = WhitePixel (TheDisplay, TheScreen);
  BgColor = BlackPixel (TheDisplay, TheScreen);
  TheWindow = XCreateSimpleWindow (TheDisplay, TheRoot,
	0, 0, dev->sizex, dev->sizey, 0, FgColor, BgColor);
  XSelectInput (TheDisplay, TheWindow, 
	ExposureMask|KeyPressMask|StructureNotifyMask);
  XMapRaised (TheDisplay, TheWindow);
  XSetStandardProperties (TheDisplay, TheWindow, APPNAME, APPNAME,
	0, NULL, 0, NULL);
  TheGC = DefaultGC (TheDisplay, TheScreen);

  do {
    XNextEvent (TheDisplay, &event);
  } while (event.type != Expose);


	dev->npages = 1;
	dev->FontWidth  = 8;
	dev->FontHeight = 8;

	for (i = 0; i < 256; ++i)
		colors[i] = NOCOLOR;

	TheColormap = DefaultColormap (TheDisplay, DefaultScreen (TheDisplay));

	st.black   = 0;	/* cannot change */
	st.white   = 1;	/* cannot change */
	dev->colors = 256;
	i = XDisplayCells (TheDisplay, TheScreen);
	if (i < 16) {
		SetPalette (st.black, C_BLACK);
		SetPalette (st.white, C_WHITE);
		st.red     =
		st.blue    =
		st.magenta = 
		st.green   =
		st.brown   =
		st.gray    =
		st.hudlow  =
		st.hudhigh =
		st.lblue   =
		st.lred    =
		st.lgray   =
		st.ground  = st.white;
	} else {
		st.red     = 2;	/* do not change! */
		st.blue    = 4;	/* do not change! */
		st.magenta = 6;	/* do not change! */
		st.green   = 3;
		st.brown   = 5;
		st.gray    = 7;
		st.hudlow  = 8;
		st.hudhigh = 9;
		st.lblue   = 10;
		st.lred    = 11;
		st.lgray   = 12;
		st.ground  = 13;
/*		st.white   = 15;	keep 15 for OR'ed white */
		SetPalette (st.black,   C_BLACK);
		SetPalette (st.white,   C_WHITE);
		SetPalette (st.red,     C_RED);
		SetPalette (st.blue,    C_BLUE);
		SetPalette (st.magenta, C_MAGENTA);
		SetPalette (st.green,   C_GREEN);
		SetPalette (st.brown,   C_BROWN);
		SetPalette (st.gray,    C_GRAY);
		SetPalette (st.hudlow,  C_LYELLOW);
		SetPalette (st.hudhigh, C_YELLOW);
		SetPalette (st.lred,    C_LIGHTRED);
		SetPalette (st.lblue,   C_SKYBLUE);
		SetPalette (st.lgray,   C_LIGHTGRAY);
		SetPalette (st.ground,  C_GRAY);
	}
	dev->colors = i;

	SetVisual (0);
	SetActive (0);

        ClearWin ();

	if (st.fname == 0)
		st.fname = strdup ("6x8.fnt");

	return (0);
}

static void FAR
Term (DEVICE *dev)		/* done */
{
	int	i;

	for (i = 0; i <256; ++i)
		if (NOCOLOR == colors[i])
			/* FreeThisColor */;

	XDestroyWindow (TheDisplay, TheWindow);
	XCloseDisplay (TheDisplay);
}

static void FAR
Flush (void)
{
	XFlush (TheDisplay);
	(void) XEventsQueued (TheDisplay, QueuedAfterFlush);
}

static void FAR
Shutters (int eye)
{
	return;
}

struct GrDriver GrX = {
	"GrX",
	0,
#if 0
	devices,
#else
	0,
#endif
	Init,
	Term,
	OpenTextWindow,
	CloseTextWindow,
	FontSet,
	TextPut,
	TextChar,
	TextColor,
	CursorMode,
	MoveTo,
	DrawTo,
	SetVisual,
	SetActive,
	0,	/* Clear */
	SetTextPos,
	PushTextPos,
	PopTextPos,
	TextClear,
	WriteMode,
	SetPalette,
	Ellipse,
	Sync,
	Flush,
	Shutters
};

#define PO	p->opt

extern int FAR
GetMouse (POINTER *p)
{
  Window root, child;
  int root_x, root_y, win_x, win_y, sizex, sizey;
  unsigned int keys_buttons;

  if (!XQueryPointer (TheDisplay, TheWindow, &root, &child, &root_x, &root_y,
     &win_x, &win_y, &keys_buttons))
    return (0);

  sizex = fmul (CW->orgx, CS->sizex);
  sizey = fmul (CW->orgy, CS->sizey);
  win_x -= sizex + CS->minx;
  win_y -= sizey + CS->miny;

  sizex = fmul (CW->maxx, CS->sizex);
  sizey = fmul (CW->maxy, CS->sizey);

  p->a[PO[3]] = muldiv (win_y, 100, sizey);		/* y */
  p->a[PO[3]] = muldiv (p->a[PO[3]]*PO[2], 10, 1+PO[7]);
  if (p->a[PO[3]] > 100)
    p->a[PO[3]] = 100;
  if (p->a[PO[3]] < -100)
    p->a[PO[3]] = -100;

  p->a[PO[1]] = muldiv (win_x, 100, sizex);		/* x */
  p->a[PO[1]] = muldiv (p->a[PO[1]]*-PO[0], 10, 1+PO[6]);
  if (p->a[PO[1]] > 100)
    p->a[PO[1]] = 100;
  if (p->a[PO[1]] < -100)
    p->a[PO[1]] = -100;

  p->b[PO[4]] += !!(keys_buttons & Button3Mask);
  p->b[PO[5]] += !!(keys_buttons & Button1Mask);

  return (0);
}

static int
GetSpecial (XKeyEvent *xkey, KeySym keysym)
{
  int k;

  switch (keysym) {
  default:
    return (-1);
  case XK_Right:
    k = K_RIGHT;
    break;
  case XK_Left:
    k = K_LEFT;
    break;
  case XK_Up:
    k = K_UP;
    break;
  case XK_Down:
    k = K_DOWN;
    break;
  case XK_Begin:
    k = K_CENTER;
    break;
  case XK_Home:
    k = K_HOME;
    break;
  case XK_End:
    k = K_END;
    break;
  case XK_Prior:
    k = K_PGUP;
    break;
  case XK_Next:
    k = K_PGDN;
    break;
  case XK_Insert:
    k = K_INS;
    break;
  case XK_F1:         /* Warning: a hack! */
  case XK_F2:
  case XK_F3:
  case XK_F4:
  case XK_F5:
  case XK_F6:
  case XK_F7:
  case XK_F8:
  case XK_F9:
  case XK_F10:
  case XK_F11:
  case XK_F12:
    k = K_F1 + (keysym - XK_F1);
    break;
  }

  k |= K_SPECIAL;

  if (xkey->state & ShiftMask)
    k |= K_SHIFT;
  if (xkey->state & ControlMask)
    k |= K_CTRL;
  if (xkey->state & (Mod1Mask|Mod5Mask))
    k |= K_ALT;

  return (k);
}

static int
GetKey (XKeyEvent *xkey)
{
	KeySym	keysym;
	char	buffer[20];
	XComposeStatus	cs;
	int	n, c;

	n = XLookupString (xkey, buffer, sizeof(buffer)-1,
		&keysym, &cs);
	if (n)
		c = 0x0ff & buffer[0];
	else
		c = GetSpecial (xkey, keysym);
	return (c);
}

static void
resize (XConfigureEvent *cf)
{
  CS->sizex = cf->width;
  CS->sizey = cf->height;
  set_screen (cf->width, cf->height);
}

extern int FAR
xkread (void)
{
	XEvent	event;
	int	n;

	for (n = -1; n < 0;) {
		if (!XEventsQueued (TheDisplay, QueuedAlready))
			return (-1);
		XNextEvent (TheDisplay, &event);
		switch (event.type) {
 		case  KeyPress:
			n = GetKey (&event.xkey);
			break;
		case ConfigureRequest:
		case ConfigureNotify:
			resize (&event.xconfigure);
			ClearWin ();
			break;
		default:
			ClearWin ();
			break;
		}
	}

	return (n);
}
