/* --------------------------------- remote.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Handler for the remote comms (very high level).
 * This module is aware of the application data structures. It uses packet
 * services to communicate with remotely participating programs.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fly.h"

/* Our comms protocol. All message numbers should be >1 and <0xff
*/

#define	RC_MAINT	0x01
#define	RC_ADD		0x02
#define	RC_STAT		0x03
#define	RC_URHIT	0x04
#define	RC_IMHIT	0x05
#define	RC_SHOOT	0x06
#define	RC_TEXT		0x07
#define	RC_TIMEREQ	0x08
#define	RC_TIMEACK	0x0a
#define	RC_ACTIVEQUERY	0x10		/* ping broadcast */
#define	RC_ACTIVEREPLY	0x11		/* ping reply */
#define	RC_NOTACTIVE	0x12		/* indicate going offline */
#define	RC_REQUEST	0x13		/* request to play */
#define	RC_REPLYPOS	0x14		/* accept player's request */
#define	RC_REPLYNEG	0x15		/* announce 'not playing' */

/* Enforce consistent byte order in message data
*/

#define PUTBYTE(b)	(*d++ = (Uchar)(b))
#define PUTWORD(w)	(t = (w), *d++ = (Uchar)(t>>8), *d++ = (Uchar)t)
#define PUTLONG(w)	(lt = (w), *d++ = (Uchar)(lt>>24), \
			 *d++ = (Uchar)(lt>>16), *d++ = (Uchar)(lt>>8), \
			 *d++ = (Uchar)lt)
#define	GETBYTE		(t = *d++)
#define	GETWORD		(t = *d++<<8, t += *d++)
#define	GETLONG		(lt = *d++, lt = (lt<<8)+*d++, lt = (lt<<8)+*d++, \
			 lt = (lt<<8)+*d++)

extern PACKET * FAR
packet_new (Ushort size)
{
	PACKET	*pack;

	if (!NEW (pack))
		return (0);

	if (0 == size)
		size = ONEPACKLEN;
	pack->size = size;
	if (!(pack->data = memory_alloc (PACKHEADLEN + size))) {
		DEL (pack);
		return (0);
	}

	return (pack);
}

extern PACKET * FAR
packet_del (PACKET *pack)
{
	memory_free (pack->data, PACKHEADLEN+pack->size);
	return (DEL (pack));
}

extern int FAR
crc (PACKET *pack)
{
	int	i, crc;
	Uchar	*p;

	p = &pack->data[PACKHEADLEN];
	crc = i = pack->length-2;
	while (i-- > 0)
		crc = (crc^(crc << 5)) ^ (int)*p++;
	return (crc);
}

static int FAR
send_packet (PACKET *pack, Uchar *d, PLAYER *player, int mode)
/*
 * mode:
 *	if player given, send to this player only.
 *	0 send to playing players
 *	1 send to known players
 *	2 broadcast to all netports
 *	3 broadcast to playing netports
*/
{
	PLAYER	*pl;
	int	t, ret;

	pack->length = (d - pack->data) - PACKHEADLEN + 2;
	if ((Uint)pack->length > pack->size) {
		MsgEPrintf (-100, "Packet %02x too long (%d)",
				pack->data[PACKHEADLEN], pack->length);
		packet_del (pack);
		return (1);
	}
	PUTWORD (crc (pack));

	if (player) {
		pack->netport = player->netport;
		pack->address = player->address;
		ret = packet_send (pack, 0);
	} else {
		if (mode == 2 && !(st.network & NET_NOBCAST))
			ret = packet_send (pack, 1);
		else if (mode == 3 && !(st.network & NET_NOBCAST))
			ret = packet_send (pack, 2);
		else {
			ret = 0;
			for (pl = 0; (pl = player_next (pl));) {
				if (mode == 0 && !(pl->flags & RMT_SEND))
					continue;
				pack->netport = pl->netport;
				pack->address = pl->address;
				if ((ret = packet_send (pack, 0)))
					break;
			}
		}
	}
	packet_del (pack);
	return (ret);
}

static int FAR
send_command (int command, PLAYER *player, int mode)
{
	PACKET	*pack;
	Uchar	*d;

	if (!(pack = packet_new (0)))
		return (1);
	d = &pack->data[PACKHEADLEN];

	PUTBYTE (command);

	return (send_packet (pack, d, player, mode));
}

static int FAR
send_stat (int command, OBJECT *p, PLAYER *player, int mode)
{
	int	t;
	long	lt;
	PACKET	*pack;
	Uchar	*d;

	if (!(pack = packet_new (0)))
		return (1);
	d = &pack->data[PACKHEADLEN];

	PUTBYTE (command);
	PUTLONG (p->id);
	PUTWORD (p->name);
	PUTWORD (p->flags);
	PUTWORD (p->gpflags);
	PUTWORD (p->color);
	PUTWORD (p->damage);
	PUTWORD (p->a[X]);
	PUTWORD (p->a[Y]);
	PUTWORD (p->a[Z]);
	PUTWORD (p->da[X]);
	PUTWORD (p->da[Y]);
	PUTWORD (p->da[Z]);
	PUTLONG (p->R[X]);
	PUTLONG (p->R[Y]);
	PUTLONG (p->R[Z]);
	PUTWORD (p->V[X]);
	PUTWORD (p->V[Y]);
	PUTWORD (p->V[Z]);
	PUTWORD (p->speed);
	PUTLONG (st.present);

	return (send_packet (pack, d, player, mode));
}

static int FAR
send_maint (int command, OBJECT *p, PLAYER *player, int mode)
{
	int	t;
	long	lt;
	PACKET	*pack;
	Uchar	*d;

	if (!(pack = packet_new (0)))
		return (1);
	d = &pack->data[PACKHEADLEN];

	PUTBYTE (command);
	PUTLONG (p->id);
	PUTWORD (p->a[X]);
	PUTWORD (p->a[Y]);
	PUTWORD (p->a[Z]);
	PUTLONG (p->R[X]);
	PUTLONG (p->R[Y]);
	PUTLONG (p->R[Z]);
	PUTWORD (p->V[X]);
	PUTWORD (p->V[Y]);
	PUTWORD (p->V[Z]);
	PUTLONG (st.present);

	return (send_packet (pack, d, player, mode));
}

extern int FAR
remote_urhit (OBJECT *p, int speed, int extent, int damaging)
{
	int	t;
	long	lt;
	PACKET	*pack;
	Uchar	*d;

	if (!(st.network & NET_INITED))
		return (1);
	if (!(pack = packet_new (0)))
		return (1);
	d = &pack->data[PACKHEADLEN];

	PUTBYTE (RC_URHIT);
	PUTLONG (p->rid);
	PUTWORD (speed);
	PUTWORD (extent);
	PUTWORD (damaging);

	return (send_packet (pack, d, p->rplayer, 0));
}

extern int FAR
remote_imhit (OBJECT *p, int seed, int speed, int extent, int damaging)
{
	int	t;
	long	lt;
	PACKET	*pack;
	Uchar	*d;

	if (!(st.network & NET_INITED))
		return (1);
	if (!(pack = packet_new (0)))
		return (1);
	d = &pack->data[PACKHEADLEN];

	PUTBYTE (RC_IMHIT);
	PUTLONG (p->id);
	PUTWORD (seed);
	PUTWORD (speed);
	PUTWORD (extent);
	PUTWORD (damaging);

	return (send_packet (pack, d, 0, 3));
}

static int FAR
send_shoot (OBJECT *p, int weapon, int n, int seed, int interval)
{
	int	t;
	long	lt;
	PACKET	*pack;
	Uchar	*d;

	if (!(pack = packet_new (0)))
		return (1);
	d = &pack->data[PACKHEADLEN];

	PUTBYTE (RC_SHOOT);
	PUTLONG (p->id);
	PUTWORD (weapon);
	PUTWORD (n);
	PUTWORD (seed);
	PUTWORD (interval);

	return (send_packet (pack, d, 0, 3));
}

static int FAR
send_playing (int command, PLAYER *player, int mode)
{
	int	t;
	long	lt;
	PACKET	*pack;
	Uchar	*d;

	if (!(pack = packet_new (0)))
		return (1);
	d = &pack->data[PACKHEADLEN];

	PUTBYTE (command);
	PUTWORD (st.ComVersion);
	PUTLONG (st.present);
	strcpy (d, st.nikname);
	d += strlen (st.nikname)+1;

	return (send_packet (pack, d, player, mode));
}

static int FAR
send_text (char *text, PLAYER *player, int mode)
{
	int	t;
	PACKET	*pack;
	Uchar	*d;
	int	len;

	if (!(pack = packet_new (0)))
		return (1);
	d = &pack->data[PACKHEADLEN];

	PUTBYTE (RC_TEXT);
	t = pack->size - 1 - 1 - 2;	/* cmd, text, eol, crc */
	if ((len = strlen (text)) > t)
		len = t;
	memcpy (d, text, len);
	d += len;
	*d++ = '\0';

	return (send_packet (pack, d, player, mode));
}

static int FAR
send_time (int command, PLAYER *player, int mode, Ulong time)
{
	long	lt;
	PACKET	*pack;
	Uchar	*d;

	if (!(pack = packet_new (0)))
		return (1);
	d = &pack->data[PACKHEADLEN];

	PUTBYTE (command);
	PUTLONG (time);

	return (send_packet (pack, d, player, mode));
}

extern int FAR
send_obj (OBJECT *p, PLAYER *player, int mode)
{
	if (!(st.network & NET_INITED))
		return (1);
	if (!p) {
		packet_send (0, 1);	/* flush all ports */
		return (0);
	}

	if (p->flags & F_NEW)
		return (send_stat (RC_ADD, p, player, mode));
	else if (p->flags & F_MOD)
		return (send_stat (RC_STAT, p, player, mode));
	else if (p->flags & F_MAINT)
		return (send_maint (RC_MAINT, p, player, mode));
	else
		return (0);
}

static int FAR
send_state (PLAYER *player, int mode)
{
	OBJECT	*p;

	for (p = CO; p; p = p->next) {
		if (p->flags & F_EXPORTED)
			send_stat (RC_ADD, p, player, mode);
	}
	return (0);
}

static void FAR
update_rtime (OBJECT *p, PLAYER *pl, PACKET *pack, Ulong rtime)
{
	p->rtime = rtime - pl->rtime;			/* object currency */
	pl->rtimeErr += (int)(p->rtime - pack->arrived);/* time diff err */
	if (pl->rtimeErr > 256) {
		++pl->rtime;
		pl->rtimeErr /= 2;
	} else if (pl->rtimeErr < -256) {
		--pl->rtime;
		pl->rtimeErr /= 2;
	}
}

/*
 * process object-specific messages from a player.
*/

static void FAR
receive_object (OBJECT *p)
{
	int	t;
	long	lt;
	PACKET	*pack, *next, *prev;
	PLAYER	*pl;
	Uchar	*d;
	int	seed, speed, extent, n, weapon, damaging, count;

	if (!(pl = p->rplayer))
		return;

	count = 0;
	for (prev = 0, pack = pl->incoming; pack; prev = pack, pack = next) {
		next = pack->next;
		d = &pack->data[PACKHEADLEN];
		switch (*d++) {		/* command */
		case RC_STAT:
			GETLONG;
			if (p->rid != lt)
				continue;
			++count;
			GETWORD;	p->name = t;
			GETWORD;	t &= ~(F_EXPORTED|F_MOD|F_DONE);
					p->flags = t | F_IMPORTED|F_ALIVE;
			GETWORD;	n = p->gpflags;
					p->gpflags = t;
			GETWORD;	p->color = t;
			GETWORD;	/*p->damage = t;*/
			GETWORD;	p->a[X] = t;
			GETWORD;	p->a[Y] = t;
			GETWORD;	p->a[Z] = t;
			GETWORD;	p->da[X] = t;
			GETWORD;	p->da[Y] = t;
			GETWORD;	p->da[Z] = t;
			GETLONG;	p->R[X] = lt;
			GETLONG;	p->R[Y] = lt;
			GETLONG;	p->R[Z] = lt;
			GETWORD;	p->V[X] = t;
			GETWORD;	p->V[Y] = t;
			GETWORD;	p->V[Z] = t;
			GETWORD;	p->speed = t;
			GETLONG;	update_rtime (p, pl, pack, (Ulong)lt);

			Mobj (p);
			p->time = FOREVER;
			if (EIM(p))
				EIM(p)->timeout = st.ObjectTimeout;

			if (p->flags & F_CC)
				p->color = st.lred;

			break;

		case RC_MAINT:
			GETLONG;
			if (p->rid != lt)
				continue;
			++count;
			GETWORD;	p->a[X] = t;
			GETWORD;	p->a[Y] = t;
			GETWORD;	p->a[Z] = t;
			GETLONG;	p->R[X] = lt;
			GETLONG;	p->R[Y] = lt;
			GETLONG;	p->R[Z] = lt;
			GETWORD;	p->V[X] = t;
			GETWORD;	p->V[Y] = t;
			GETWORD;	p->V[Z] = t;
			GETLONG;	update_rtime (p, pl, pack, (Ulong)lt);

			Mobj (p);
			p->da[X] = p->da[Y] = p->da[Z] = 0;
			p->speed = ihypot3d (p->V);
			p->flags |= F_ALIVE;
			if (EIM(p))
				EIM(p)->timeout = st.ObjectTimeout;
			break;

		case RC_IMHIT:
			GETLONG;
			if (p->rid != lt)
				continue;
			++count;
			GETWORD;	seed = t;
			GETWORD;	speed = t;
			GETWORD;	extent = t;
			GETWORD;	damaging = t;
			object_hit (p, seed, speed, extent, damaging);
			break;

		case RC_SHOOT:
			GETLONG;
			if (p->rid != lt)
				continue;
			++count;
			GETWORD;	weapon = t;
			GETWORD;	n = t;
			GETWORD;	seed = t;
			GETWORD;	speed = t;	/* interval */
			shoot (p, weapon, n, seed, speed);
			break;

		default:
			continue;
		}
		if (prev)
			prev->next = next;
		else
			pl->incoming = next;
		packet_del (pack);
		pack = prev;		/* prepare for 'prev = pack' */
	}
	pl->tail = prev;
	if (count)
		pl->timeout = st.PlayerTimeout;
}

/*
 * process general messages from a player.
*/

static void FAR
receive_player (PLAYER *pl, int mode)
{
	int	t;
	long	lt;
	PACKET	*pack, *next, *prev;
	OBJECT	*p;
	Uchar	*d;
	int	speed, extent, count, damaging;
	long	rid;
	char	msg[LADDRESS*2+1], *m;

	count = 0;
	for (prev = 0, pack = pl->incoming; pack; prev = pack, pack = next) {
		next = pack->next;
		d = &pack->data[PACKHEADLEN];
		switch (*d++) {		/* command */
		case RC_URHIT:
			++count;
			if (!(pl->flags & RMT_RECEIVE))
				break;
			GETLONG;
			for (p = CO; p; p = p->next)
				if ((p->flags & F_EXPORTED) && p->id == lt)
					break;
			if (p) {
				GETWORD;	speed = t;
				GETWORD;	extent = t;
				GETWORD;	damaging = t;
				object_hit (p, rand(), speed, extent, damaging);
			}
			break;

		case RC_STAT:
			if (!mode)
				continue;
			++count;
			if (!(pl->flags & RMT_RECEIVE))
				break;
			++st.stats[50];		/* was lost, now found! */
			goto refresh;
		case RC_ADD:
			++count;
			if (!(pl->flags & RMT_RECEIVE))
				break;
refresh:
			GETLONG;	rid = lt;
			GETWORD;

			if (!(p = create_object (t, 0)))
				break;

			p->rplayer = pl;
			p->rid = rid;
			GETWORD;	t &= ~(F_EXPORTED|F_MOD|F_DONE);
					p->flags = t | F_IMPORTED|F_ALIVE;
			GETWORD;	p->gpflags = t;
			GETWORD;	p->color = t;
			GETWORD;	p->damage = t;
			GETWORD;	p->a[X] = t;
			GETWORD;	p->a[Y] = t;
			GETWORD;	p->a[Z] = t;
			GETWORD;	p->da[X] = t;
			GETWORD;	p->da[Y] = t;
			GETWORD;	p->da[Z] = t;
			GETLONG;	p->R[X] = lt;
			GETLONG;	p->R[Y] = lt;
			GETLONG;	p->R[Z] = lt;
			GETWORD;	p->V[X] = t;
			GETWORD;	p->V[Y] = t;
			GETWORD;	p->V[Z] = t;
			GETWORD;	p->speed = t;
			GETLONG;	update_rtime (p, pl, pack, (Ulong)lt);

			Mobj (p);
			p->time = FOREVER;
			if (NEW (EIM(p))) {
				p->e_type = ET_IMPORTED;
				EIM(p)->timeout = st.ObjectTimeout;
			}
#if 1
			if (p->flags & F_CC)
				p->color = st.lred;
#endif
			break;

		case RC_ACTIVEQUERY:			/* idle -> active */
			GETWORD;			/* version */
			if ((Uint)t != st.ComVersion) {
				MsgWPrintf (100, "ComVer %u: %s",
					(Uint)t, pl->name);
				break;
			}
			send_playing (RC_ACTIVEREPLY, pl, 0);
			goto common_playing;

		case RC_ACTIVEREPLY:			/* idle -> active */
			GETWORD;			/* version */
			if ((Uint)t != st.ComVersion)
				break;
common_playing:
			GETLONG;	pl->rtime = lt - pack->arrived;
					pl->rtimeErr = 0L;
			++count;
			if (strlen (d) < LNAME-1)
				strcpy (pl->name, d);
			if (pl->flags & (RMT_NOTIDLE & ~RMT_ACTIVE))
				break;
			MsgWPrintf (100, "Active: %s", pl->name);
			for (m = msg, t = 0; t < LADDRESS; ++t) {
				sprintf (m, "%02x", pl->address[t]);
				m += 2;
			}
			MsgPrintf (100, "Addr:   %s", msg);
			break;

		case RC_NOTACTIVE:			/* active -> idle */
			++count;
			LogPrintf ("%s ", Tm->Ctime ());
			MsgWPrintf (-100, "Gone: %s", pl->name);
			player_remove (pl);
			pl->flags &= ~RMT_NOTIDLE;
			break;

		case RC_REQUEST:			/* active -> pend */
			++count;
			MsgPrintf (100, "Asking: %s", pl->name);
			if (pl->flags & RMT_NOTIDLE) {
				player_remove (pl);
				pl->flags &= ~RMT_NOTIDLE;
			}
			pl->flags |= RMT_PENDBOSS;
			if (st.network & NET_AUTOACCEPT)
				remote_reply (pl, 1);
			else if (st.network & NET_AUTODECLINE)
				remote_reply (pl, 0);
			else
				MsgWPrintf (100, "Pending: %s", pl->name);
			break;

		case RC_REPLYPOS:		/* pend -> play */
			++count;
			if (!(pl->flags & RMT_PEND))
				break;
			if (pl->flags & RMT_PENDREPLY)
				send_command (RC_REPLYPOS, pl, 0);
			pl->flags &= ~RMT_PEND;
			pl->flags |= RMT_PLAYING;
			netport_count (pl, 1);
			send_state (pl, 0);
			LogPrintf ("%s ", Tm->Ctime ());
			MsgWPrintf (-100, "Joined: %s", pl->name);
			break;

		case RC_REPLYNEG:		/* pend/playing -> active */
			++count;
			if (pl->flags & RMT_PEND)
				MsgWPrintf (100, "Declined: %s", pl->name);
			else if (pl->flags & RMT_PLAYING) {
				LogPrintf ("%s ", Tm->Ctime ());
				MsgWPrintf (-100, "Not Playing: %s", pl->name);
			}
			player_remove (pl);
			pl->flags &= ~RMT_NOTIDLE;
			pl->flags |= RMT_ACTIVE;
			break;

		case RC_TEXT:
			++count;
			if (strlen (d) > 80)
				break;
			MsgWPrintf (100, "%s says:", pl->name);
			MsgWPrintf (100, "  %s", d);
			break;

		case RC_TIMEREQ:
			++count;
			GETLONG;			/* remote send time */
			send_time (RC_TIMEACK, pl, 0, lt);
			break;

		case RC_TIMEACK:
			++count;
			GETLONG;			/* my send time */
			MsgPrintf (100, "%s time: %lu",
				pl->name, Tm->Milli () - lt);
			break;

		default:
			continue;
		}
		if (prev)
			prev->next = next;
		else
			pl->incoming = next;
		packet_del (pack);
		pack = prev;		/* prepare for 'prev = pack' */
	}
	pl->tail = prev;
	if (count)
		pl->timeout = st.PlayerTimeout;
}

extern void FAR
remote_receive (OBJECT *obj)
{
	PLAYER	*pl;

	if (!(st.network & NET_INITED))
		return;

	if (obj)
		receive_object (obj);
	else {
		netports_receive ();
		for (pl = 0; (pl = player_next (pl));)
			receive_player (pl, 0);
	}
}

extern void FAR
remote_refresh (void)
{
	PLAYER	*pl;

	if (!(st.network & NET_INITED))
		return;

	for (pl = 0; (pl = player_next (pl));)
		receive_player (pl, 1);
}

extern void FAR
remote_play (PLAYER *pl)
{
	if (pl->flags & RMT_PLAYING) {
		MsgPrintf (100, "already playing");
		return;
	}
	send_command (RC_REQUEST, pl, 0);
	pl->flags &= ~RMT_NOTIDLE;
	pl->flags |= RMT_PENDREPLY;
	pl->timeout = st.PlayerTimeout;
}

extern void FAR
remote_reply (PLAYER *pl, int reply)
{
	if (!(pl->flags & RMT_PENDBOSS)) {
		MsgPrintf (100, "not pending");
		return;
	}
	pl->flags &= ~RMT_PEND;
	if (reply) {
		MsgPrintf (100, "Accepting: %s", pl->name);
		pl->flags |= RMT_PENDCONFIRM;
		send_command (RC_REPLYPOS, pl, 0);
	} else {
		MsgPrintf (100, "Declining: %s", pl->name);
		send_command (RC_REPLYNEG, pl, 0);
	}
}

extern void FAR
remote_noplay (PLAYER *pl, int mode)
{
	if (st.network & NET_INITED)
		send_command (RC_REPLYNEG, pl, mode);
}

extern void FAR
remote_ping (void)
{
	if (st.network & NET_INITED)
		send_playing (RC_ACTIVEQUERY, 0, 2);
}

extern void FAR
remote_shoot (OBJECT *p, int weapon, int n, int seed, int interval)
{
	if (st.network & NET_INITED)
		send_shoot (p, weapon, n, seed, interval);
}

extern void FAR
remote_msg (char *text, PLAYER *pl, int mode)
{
	if (st.network & NET_INITED)
		send_text (text, pl, mode);
}

extern void FAR
remote_time (PLAYER *pl, int mode)
{
	if (st.network & NET_INITED)
		send_time (RC_TIMEREQ, pl, mode, Tm->Milli ());
}

extern int FAR
remote_init (void)
{
	if (!(st.network & NET_ON))
		return (0);

	MsgPrintf (-100, "Net Ver  %u", (int)st.ComVersion);

	if (netports_init ())
		return (1);

	if (players_init ())
		return (1);
	st.network |= NET_INITED;
	return (0);
}

extern void FAR
remote_term (void)
{
	int	i;

	if (st.network & NET_INITED) {
		send_command (RC_NOTACTIVE, 0, 2);
		players_term ();
	}
	netports_term ();

	for (i = 0; i < 10; ++i)
		if (st.stats[i])
			LogPrintf ("%2u) %ld\n", i, st.stats[i]);
	for (i = 50; i < 60; ++i)
		if (st.stats[i])
			LogPrintf ("%2u) %ld\n", i, st.stats[i]);
	if (st.stats[21])
		LogPrintf ("in  av %lu max %lu\n",
			st.stats[22]/st.stats[20], st.stats[21]);
	if (st.stats[24])
		LogPrintf ("out av %lu max %lu\n",
			st.stats[25]/st.stats[23], st.stats[24]);
	st.network &= ~NET_INITED;
}
