/* --------------------------------- stick.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Handler for the joy-stick as a pointing device.
*/

#include <stdlib.h>
#include <dos.h>
#include <conio.h>

#include "fly.h"

/* specify whether to use the DOS int 0x15 call */
#define USEINT21 0

/* if not using USEINT21 then this selects timed or just counting
   NOTE: the timed does not seem to work */
#define COUNT_TIME	1

#define PO	p->opt
#define	BPORT	0x0001

static int FAR
read (POINTER *p)
/*
 * Read joy-stick. Values are in the range 0...200.
 */
{
#if USEINT21    	
	int		px, py;
	unsigned int	c0, x, y, minx, miny;
	int		b1, b2;

    	minx = miny = 0x7fff;

	/* read joystick values */
	_asm	mov ax, 0x8400;
	_asm	mov dx, 0x0001;
	_asm	int 0x15;
	_asm	and ax, 0x00ff;
	_asm	and dx, 0x00ff;
	_asm	mov x, ax;
	_asm	mov y, bx;

	/* read joystick buttons */
	_asm	mov ax, 0x8000;
	_asm	mov dx, 0x0000
	_asm	int 0x15
	_asm	shr ax, 4
	_asm	not ax
	_asm	and ax, 0x000f
	_asm	mov c0, ax
	
	b1 = !(c0 & 0x01);	/* right button */
	b2 = !(c0 & 0x02);	/* left button */

	if (minx > x)
		minx = x;
	if (miny > y)
		miny = y;

#else
	register int	cc;
	register char	c0, c1;
	int		px, py;
	unsigned int	t, x, y, minx, miny;
	int		ntimes, b1, b2;

    	minx = miny = 0x7fff;
    	ntimes = PO[8] ? 2 : 1;
    	for (; ntimes > 0; --ntimes) {	/* get min of two readings? */
		cc = 0;
#if !COUNT_TIME
		t = Tm->Hires ();
#endif
		outp (0x201, 0);	/* any value */
		if (p->control->flags & BPORT) {
			c0 = 0x0c;
			do {
				c1 = (char)inp (0x201);
				c1 ^= c0;
				if (c1 & 0x08)
#if COUNT_TIME
					y = cc;
#else
					y = Tm->Hires ();
#endif
				if (c1 & 0x04)
#if COUNT_TIME
					x = cc;
#else
					x = Tm->Hires ();
#endif
				c0 ^= c1;
			} while (++cc && (c0 & 0x0c));
			b1 = !(c0 & 0x40);	/* right button */
			b2 = !(c0 & 0x80);	/* left button */
		} else {
			c0 = 0x03;
			do {
				c1 = (char)inp (0x201);
				c1 ^= c0;
				if (c1 & 0x02)
#if COUNT_TIME
					y = cc;
#else
					y = Tm->Hires ();
#endif
				if (c1 & 0x01)
#if COUNT_TIME
					x = cc;
#else
					x = Tm->Hires ();
#endif
				c0 ^= c1;
			} while (++cc && (c0 & 0x03));
			b1 = !(c0 & 0x10);	/* right button */
			b2 = !(c0 & 0x20);	/* left button */
		}
		if (!cc)
			return (1);
#if !COUNT_TIME
		x -= t;
		y -= t;
#endif
		if (minx > x)
			minx = x;
		if (miny > y)
			miny = y;
    	}
#endif
	p->b[PO[4]] += b1;		/* right button */
	p->b[PO[5]] += b2;		/* left button */

	px = PO[1];
	py = PO[3];

	p->a[px] = minx;
	p->a[py] = miny;

	if (p->c[px] == 0 || p->c[py] == 0) /* calibrating */
		return (0);

#define	REF	100		/* expected full range */
#define	DULL	20		/* dull center */
#define	EDGE	10		/* dull edge */

#define	MDULL	(REF+DULL)
#define	FULL	(MDULL+EDGE)

	p->a[px] -= p->c[px];		/* center */
	p->a[px] *= -PO[0];		/* orientation */
	if (p->a[px] > p->c[px])	/* scale */
		p->a[px] = REF;
	else if (p->a[px] < -p->c[px])
		p->a[px] = -REF;
	else {
		p->a[px] = muldiv (p->a[px], FULL, p->c[px]);
		if      (p->a[px] > MDULL)
			p->a[px] = REF;
		else if (p->a[px] >= DULL)
			p->a[px] -= DULL;
		else if (p->a[px] < -MDULL)
			p->a[px] = -REF;
		else if (p->a[px] <= -DULL)
			p->a[px] += DULL;
		else
			p->a[px] = 0;
	}

	p->a[py] -= p->c[py];		/* center */
	p->a[py] *= PO[2];		/* orientation */
	if (p->a[py] > p->c[py])	/* scale */
		p->a[py] = REF;
	else if (p->a[py] < -p->c[py])
		p->a[py] = -REF;
	else {
		p->a[py] = muldiv (p->a[py], FULL, p->c[py]);
		if      (p->a[py] > MDULL)
			p->a[py] = REF;
		else if (p->a[py] >= DULL)
			p->a[py] -= DULL;
		else if (p->a[py] < -MDULL)
			p->a[py] = -REF;
		else if (p->a[py] <= -DULL)
			p->a[py] += DULL;
		else
			p->a[py] = 0;
	}
	return (0);
}

static int FAR
cal (POINTER *p)
/*
 * Calibrate joy-stick. Paddle must be at center!
 *
 */
{
	p->c[PO[1]] = p->c[PO[3]] = 0;	/* indicate 'calibrating' */
	if (read (p))
		return (1);
	p->c[PO[1]] = p->a[PO[1]];
	p->c[PO[3]] = p->a[PO[3]];

	p->a[PO[1]] = p->a[PO[3]] = 0;
	p->l[PO[1]] = p->l[PO[3]] = 0;
	if (p->c[PO[1]] == 0 || p->c[PO[3]] == 0)
		return (1);
	return (0);
}

static void FAR
term (POINTER *p)
{
       	p = p;
	return;
}

static int FAR
center (POINTER *p)
{
	p->a[PO[1]] = p->a[PO[3]] = 0;
	p->l[PO[1]] = p->l[PO[3]] = 0;

	return (0);
}

extern struct PtrDriver PtrAstick = {
	"ASTICK",
	0,
	cal,			/* init */
	term,
	cal,
	center,
	read,
	std_key
};

extern struct PtrDriver PtrBstick = {
	"BSTICK",
	BPORT,
	cal,			/* init */
	term,
	cal,
	center,
	read,
	std_key
};

