
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;

import org.webmacro.util.java2.*;
import java.util.*;
import java.io.*;
import org.webmacro.util.*;

/**
  * This directive is used to iterate through the members of a list. 
  * <p>
  * It takes an argument of type Variable and iterates through 
  * each of its members. Each time through the loop a scalar with the 
  * same name of the list is defined, and takes on the value of the list 
  * for that element. The scalar will have the same type as the value of
  * the list element, so if the list contains hashtables, referencing the
  * members of the hashtable will work as expected.
  */
final class ForeachDirective implements Macro
{


   final private static boolean _debug = false; 

   /**
     * This is a reference to the list we iterate through
     */
   final private Object _list;

   /**
     * This is the block that is output each time through the iteration
     */
   final private Block _body;

   /**
     * This is the variable reference used as the iterator
     */
   final private Variable _iterVar;

   /**
     * Whether or not our list var is a macro
     */
   final private boolean _macro;

   /**
     *  Constuctor for ForeachDirective.
     */
   ForeachDirective(Object list, Variable iterVar, Block body) {
      _list = list; 
      _iterVar = iterVar;
      _body = body;
      _macro = (list instanceof Macro);
   }


   /**
     * Parse the directive. Expect format "foreach $person in $customer"
     * etc., Block has already eaten the '#' that implies a directive
     * <p>
     * @return ForeachDirective
     * @exception IOException if there is an error from the parseTool
     * @exception ParseException if an unrecoverable parse error occurs
     */
   static public final Object parse(ParseTool in) 
      throws IOException, ParseException
   {

      Object  list;
      Block body;
      Variable iterVar;

      // look for "foreach" key word  
      if ((in.ttype !=  in.TT_WORD) && (!in.sval.equals("foreach"))) {
         if (_debug) {
            Engine.log.debug("ForeachDirective.parse: " 
                  + in.sval + " unrecognized.");
         }
         return null;
      } 
     
      in.nextToken(); // advance token by one  
      if (! in.parseSpaces()) {  // expects to be at space advance till no space
         if (_debug) {
	    Engine.log.debug("Foreach ttype:" + (char)in.ttype);
	    Engine.log.debug("Foreach sval:" + in.sval);
         }
         throw new ParseException(in,
               "expected spaces after keyword, got: " + (char) in.ttype);
      }

      if (in.ttype != '$'){ // do not eat $ sign check for only
         throw new ParseException(in,
               "expected $ after foreach key word, got: " + (char)in.ttype);
      }
    
      // get name of iter variable
      iterVar = (Variable) Variable.parse(in);
      if (iterVar == null) {
         throw new ParseException(in,
               "expected TT_WORD, got: " + (char)in.ttype);
      }    

      if (! in.parseSpaces()) {  // expects to be at space adv till no space 
         throw new ParseException(in,
               "expected spaces after keyword, got: " + (char)in.ttype);
      }

      // check for a word "in"
      if ((in.ttype !=  in.TT_WORD) && (!in.sval.equals("in"))) {
         throw new ParseException(in,
               "expected in after foreach key word, got: " + (char)in.ttype);
      }  

      in.nextToken(); // advance token into space

      if (! in.parseSpaces()) {  // expects to be at space adv till no space 
         throw new ParseException(in,
               "(3) expected spaces after keyword, got: " + (char)in.ttype);
      }

      // parse list of items
      list = Term.parse(in);
      if (list == null) {
         throw new ParseException(in,
               "expected TT_WORD, got: " + (char)in.ttype);
      }    

      if (! in.parseWhitespace()) { // expects space or newline eat till not 
         throw new ParseException(in,
               "expected ' ' after iterating variable name: got " + (char) in.ttype);
      }

      // do not eat { check only
      if (in.ttype != '{') {
         throw new ParseException(in,
               "Block open brace required got: " + (char) in.ttype);
      }

      // parse block
      body = (Block) Block.parse(in);
      if (body == null) {
         throw new ParseException(in,
            "Variable  requires a block");
      }

      return new ForeachDirective(list, iterVar, body);
   }

   /**
     * Interpret the directive and write it out
     * <p>
     * @exception InvalidContextException if required data was missing from context
     * @exception IOException if we could not successfully write to out
     */
   public void write(Writer out, Object context) 
      throws InvalidContextException, IOException
   {
      // XXX: This method no longer protects values in a loop, they can
      // now clobber values outside the loop:
      // Map listMap = new HashMap();
      // listMap.include(context);

      Object list = _list;
      if (_macro) {
         while (list instanceof Macro) {
            list = ((Macro) list).evaluate(context);
         }
      }

      Iterator iter;
      try {
         iter = PropertyOperator.getIterator(list);
      } catch (Exception e) {
         throw new InvalidContextException("The object used as the list of values in a foreach statement must have some way of returning a list type, or be a list type itself. See the documentation for PropertyOperator.getIterator() for more details. No such property was found on the supplied object: " + list);
      }
      Object listItem;

      // deals with an empty list substitute appropriate variables and 
      // prints block out once
      if (iter == null) {
	 listItem = "<!--\n " +  _list + ": is empty \n-->";   
         try {
            _iterVar.setValue(context, listItem);
         } catch (InvalidContextException e) {
            Engine.log.exception(e);
            Engine.log.error("Unable to resolve list" + _list);
            out.write("<!--\n Unable to resolve list " + _list + " \n-->");
         }
         _body.write(out, context);

      // iterates through all items including null ones which will
      // print error messages
      } else {
         while(iter.hasNext()) {
            if ((listItem = iter.next()) == null) {
	       listItem = "<!--\n " +  _list + ": contained a null item \n-->";   
	    }
            try {
               _iterVar.setValue(context, listItem);
               _body.write(out, context);
            } catch (InvalidContextException e) {
               Engine.log.exception(e);
               Engine.log.error("unable to set a list item of list: " + _list); 
               out.write("<!--\n Unable to resolve list " + _list + " \n-->");
            }
         }
      }
   }

   /**
     * Interpret the foreach directive by looking up the list reference
     * in the supplied context and iterating through it. Return the 
     * result as a string.
     * @exception InvalidContextException is required data is missing
     */ 
   public Object evaluate(Object context)
      throws InvalidContextException
   {
      try {
         StringWriter sw = new SizedStringWriter(512);
         write(sw,context);
         return sw.toString();
      } catch (IOException e) {
         Engine.log.exception(e);
         Engine.log.error("evaluate got IO exception on write to StringWriter");
         return "";
      }
   }  


   /**
     * Test harness
     */
   public static void main(String arg[]) {
   
      Log.setLevel(Log.DEBUG);
      Log.traceExceptions(true);
      try {
         Map c = new HashMap();

         Vector customers = new Vector();
         customers.addElement("Dave");
         customers.addElement("Susan");
         customers.addElement("yuenping");
         customers.addElement("justin");
         customers.addElement("anjli");
         customers.addElement("Mark");

         c.put("customers",customers);

         ParseTool in = new ParseTool(new File("examples","justList.wm"));
         in.nextToken();
         in.nextToken();
	 Writer out = new FileWriter(
               new File("examples","justList.html"));
         ForeachDirective a = (ForeachDirective) ForeachDirective.parse(in);
         System.out.println("Converting justList.wm to justList.html");
	 a.write(out, c);
         out.close();
         
      } catch (Exception e) {
         e.printStackTrace();
      }
 
   }

}


