package com.tolstoy.imagemeister;

import java.io.File;
import java.awt.*;
import java.awt.image.*;
import com.jconfig.*;

/**
The GUI portion of ImageMeister.

This object is owned by a BrowserDisplayOwnerI object, which is called when the user clicks the buttons of the
GUI or the files/folders being displayed.

<P>
This object owns a GridScrollerPanel object which displays the files/folders. It also implements the GSPCallback
interface used by the GridScrollerPanel. This last object calls the methods of the GSPCallback interface to display
the file/folder icons, and also when the user double-clicks one of the file/folder icons.

<P>
This object owns a ButtonPanel, which contains the control buttons (quit, about, etc.)

<P>
This object contains an array of Image objects, one Image for each file/folder icon. These Image objects are
only created when needed. For instance, if the user navigates to a folder which contains 500 items, and only
the first 100 are visible in the GridScrollerPanel, these first 100 Images will be created when the display
is updated. However, the remaining 400 will only be created if the user scrolls down.

Copyright (c) 1998-2002 Samizdat Productions. All Rights Reserved. ImageMeister is a Trademark of Samizdat Productions.
*/

class BrowserDisplay extends Frame implements GSPCallback, ButtonPanelOwnerI {
	private static final String copyright="ImageMeister Copyright (c) 1998-2002 Samizdat Productions. All Rights Reserved. ImageMeister is a Trademark of Samizdat Productions.";

	private static final String		kFrameTitle = "ImageMeister";
	private static final int		kGridWidth = 108;
	private static final int		kGridHeight = 64;

	private static final int		kIconWidth = 32;
	private static final int		kIconHeight = 32;

	private static final int		kIconOffsetX = ( kGridWidth - kIconWidth ) / 2;
	private static final int		kIconOffsetY = 6;
//	private static Font				theFont = new Font( "Helvetica", Font.PLAIN, 10 );

	private BrowserDisplayOwnerI	owner;
	private GridScrollerPanel		theGSP;
	private ButtonPanel				theButtonPanel;
	private DiskObject				theDiskObjs[];
	private	Image					theImages[];
	private boolean					selectFlags[];
	private int						smIconData[], smIconW, smIconH,
									lgIconData[], lgIconW, lgIconH;

	public BrowserDisplay( String imageBase, BrowserDisplayOwnerI onr ) {
		super( kFrameTitle );
		owner = onr;

		theGSP = new GridScrollerPanel( this, kGridWidth, kGridHeight );

		theButtonPanel = new ButtonPanel( this, imageBase );
		add( "North", theButtonPanel );
		add( "Center", theGSP );
		reshape( 0, 0, 580, 300 );

		theDiskObjs = null;
		theImages = null;
		selectFlags = null;
	}

	public void setVisibility( boolean bState ) {
		if ( bState )
			show();
		else
			hide();
	}

/**
From the GSPCallback interface.
Tell our owner the user double-clicked a file/folder icon.
*/

	public void itemDoubleClicked( int which ) {
		owner.browserItemClicked( which );
	}

/**
From the GSPCallback interface.
Draw the file/folder icon and the name of the object.
*/

	public void paintItem( Graphics g, int which, int x, int y, int w, int h, boolean bSelected ) {
		FontMetrics		metrics;
		String			name;
		int				nameWidth;

		if ( bSelected ) {
			g.drawRect( x + 1, y + 1, w - 2, h - 2 );
			g.drawRect( x + 2, y + 2, w - 4, h - 4 );
		}

		showLargeIcon( which, g, x + kIconOffsetX, y + kIconOffsetY );
//		g.setFont( theFont );
		name = theDiskObjs[ which ].getDisplayName();
		if ( name == null )
			return;

		metrics = g.getFontMetrics();
		nameWidth = metrics.stringWidth( name );
		if ( nameWidth > ( kGridWidth - 12 ) ) {
			name = shortenStringToFit( name, metrics, kGridWidth - 12 );
			nameWidth = metrics.stringWidth( name );
		}
	
		g.drawString( name, x + ( kGridWidth - nameWidth ) / 2, y + h - 12 );
	}

/**
From the GSPCallback interface.
Tell our owner the user pressed a key.
*/

	public boolean handleKeyPress( Event e, int curChic ) {
		return owner.browserKeyPressed( curChic, e.key );
	}

	public DiskObject getItem( int which ) {
		if ( theDiskObjs == null || which < 0 || which >= theDiskObjs.length )
			return null;
		
		return theDiskObjs[ which ];
	}

	public int getSelectedItemIndex() {
		return theGSP.getCurChic();
	}

	public DiskObject getSelectedItem() {
		int			curChic;
		
		curChic = theGSP.getCurChic();
		if ( curChic < 0 || theDiskObjs == null || curChic >= theDiskObjs.length )
			return null;

		return theDiskObjs[ curChic ];
	}

/**
Set the currently displayed files/folders.
Flush the Images currently in 'theImages',
and cause the GridScrollerPanel the call us back to draw the new files/folders.
*/

	public void setItems( DiskObject dObjs[] ) {
		int			i, len;

		if ( dObjs == null )
			return;

		if ( theImages != null ) {
			len = theImages.length;
			for ( i = 0; i < len; i++ ) {
				if ( theImages[ i ] != null ) {
					theImages[ i ].flush();
					theImages[ i ] = null;
				}
			}
			theImages = null;
		}
		
		len = dObjs.length;
		theImages = new Image[ len ];
		for ( i = 0; i < len; i++ )
			theImages[ i ] = null;

		theDiskObjs = dObjs;
		theGSP.setNumItems( dObjs.length );
		theGSP.repaint();
	}

	public Frame getMainFrame() {
		return this;
	}

	public void setItemText( int which, String text ) {
		theButtonPanel.setFileString( text );
	}

	public boolean getCheckboxValue( int which ) {
		if ( which == kCheckboxShowAllFiles )
			return theButtonPanel.getCheckboxValue( ButtonPanelI.kCheckboxShowAllFiles );
		else if ( which == kCheckboxNewWindow )
			return theButtonPanel.getCheckboxValue( ButtonPanelI.kCheckboxNewWindow );

		return false;
	}

	public void setCheckboxValue( int which, boolean state ) {
		if ( which == kCheckboxShowAllFiles )
			theButtonPanel.setCheckboxValue( ButtonPanelI.kCheckboxShowAllFiles, state );
		else if ( which == kCheckboxNewWindow )
			theButtonPanel.setCheckboxValue( ButtonPanelI.kCheckboxNewWindow, state );
	}

/**
From the ButtonPanelOwnerI interface.
*/

	public void buttonClicked( int which ) {
		switch ( which ) {
			case ButtonPanelOwnerI.kButtonQuit:
				owner.browserButtonClicked( BrowserDisplayOwnerI.kButtonQuit );
			break;
			case ButtonPanelOwnerI.kButtonAbout:
				owner.browserButtonClicked( BrowserDisplayOwnerI.kButtonAbout );
			break;
			case ButtonPanelOwnerI.kButtonInfo:
				owner.browserButtonClicked( BrowserDisplayOwnerI.kButtonInfo );
			break;
			case ButtonPanelOwnerI.kButtonShow:
				owner.browserButtonClicked( BrowserDisplayOwnerI.kButtonShow );
			break;
			case ButtonPanelOwnerI.kButtonUpALevel:
				owner.browserButtonClicked( BrowserDisplayOwnerI.kButtonUpALevel );
			break;
			case ButtonPanelOwnerI.kCheckboxShowAllFiles:
				owner.browserButtonClicked( BrowserDisplayOwnerI.kCheckboxShowAllFiles );
			break;
			case ButtonPanelOwnerI.kCheckboxNewWindow:
				owner.browserButtonClicked( BrowserDisplayOwnerI.kCheckboxNewWindow );
			break;
		}
	}

	public boolean handleEvent( Event ev ) {
		if ( ev.id == Event.WINDOW_DESTROY ) {
			owner.browserCloseButtonClicked();
			return true;
		}

		return false;
	}

/**
Adds an ellipsis to long file names.
*/

	private String shortenStringToFit( String in, FontMetrics metrics, int width ) {
		int			len;

		len = in.length();
		while ( len > 4 ) {
			len -= 2;
			in = in.substring( 0, len );
			if ( metrics.stringWidth( in ) < width )
				break;
		}

		return in + "...";
	}

	void showLargeIcon( int which, Graphics g, int x, int y ) {
		DiskObject				diskObject;
		MemoryImageSource		mis;
		Image					image;
		IconBundle				ib;
		int						theErr;

		if ( theImages == null )
			return;

		if ( theImages[ which ] != null ) {
			g.drawImage( theImages[ which ], x, y, null );
			return;
		}

		diskObject = theDiskObjs[ which ];
		ib = diskObject.getIconBundle();
		if ( ib == null ) {
			Trace.println( "can't get icon bundle from " + diskObject.getName() );
			return;
		}

		if ( lgIconData == null ||
		lgIconW != ib.getIconWidth( IconBundle.ICON_LARGE ) ||
		lgIconH != ib.getIconHeight( IconBundle.ICON_LARGE ) ) {
			lgIconW = ib.getIconWidth( IconBundle.ICON_LARGE );
			lgIconH = ib.getIconHeight( IconBundle.ICON_LARGE );
			lgIconData = new int[ lgIconW * lgIconH ];
		}

		theErr = ib.getIcon( IconBundle.ICON_LARGE, IconBundle.ICON_CHANGE_NONE, IconBundle.ICON_ALIGN_NONE, lgIconData );

		if ( theErr != ErrCodes.ERROR_NONE ) {
			Trace.println( "got err=" + theErr + " from icon bundle for " + diskObject.getName() );
			return;
		}

		mis = new MemoryImageSource( lgIconW, lgIconH, lgIconData, 0, lgIconW );

		theImages[ which ] = createImage( mis );

		g.drawImage( theImages[ which ], x, y, null );
	}
}

