/** \file WO_Cache.h
    Generic cache.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_WO_CACHE
#define H_WO_CACHE

#pragma comment(lib, "libWO.lib")

#include <DS_Hashtable.h>
#include <CON_Resource.h>

/** Cache internal data structure. */
template<class T>
class Name_Containable : public SimpleString_Containable
{
public:
  Name_Containable(String& Name, T* Obj) : 
    SimpleString_Containable(Name),
    m_Obj(Obj)
  {}
  ~Name_Containable() { if (m_Obj!=NULL) delete m_Obj; }

  T* m_Obj;
};

class Interface_Name_Containable : public SimpleString_Containable
{
public:
  Interface_Name_Containable(String& Name, Interface* Obj) : 
    SimpleString_Containable(Name),
    m_Obj(Obj)
  {}
  ~Interface_Name_Containable() { if (m_Obj!=NULL) m_Obj->release(); }

  Interface* m_Obj;
};


/** Generic cache.  Each data object has a name which is its load key.
    The pure virtual loadNew() method must be overridden to provide specific
    object loading behaviour.
*/
template<class T>
class Cache
{
public:
  /** Construct a cache, and attach a resource stream that will be used to load new objects from. */
  Cache(ResourceStream& Resources) : 
    m_Resources(Resources), 
    m_Objects(Container::DESTRUCTIVE)
  {}

  virtual ~Cache() {}
  
  /** Load an object from the cache. */
  virtual T* load(char* name)
  {
    String Name=name;
    SimpleString_Containable SSC(Name);
    Name_Containable<T>* TNC=(Name_Containable<T>*)m_Objects.get(&SSC);
    if (TNC!=NULL) return TNC->m_Obj;
    T* t=loadNew(Name);
    m_Objects.put(new Name_Containable<T>(Name,t));
    return t;
  }

  /** Returns the resource stream associated with this cache. */
  ResourceStream& getResourceStream() 
  {
    return m_Resources;
  }

protected:
  virtual T* loadNew(String& Name) = 0;
  ResourceStream& m_Resources;
  Hashtable       m_Objects;
};

class InterfaceCache
{
public:
  InterfaceCache(ResourceStream& Resources) : 
    m_Resources(Resources), 
    m_Interfaces(Container::DESTRUCTIVE)
  {}

  virtual ~InterfaceCache() {}

  /** Load an interface from the cache. */
  virtual Interface* load(char* name)
  {
    String Name=name;
    SimpleString_Containable SSC(Name);
    Interface_Name_Containable* INC=(Interface_Name_Containable*)m_Interfaces.get(&SSC);
    if (INC!=NULL) return INC->m_Obj;
    Interface* I=loadNew(Name);
    m_Interfaces.put(new Interface_Name_Containable(Name,I));
    return I;
  }

  /** Returns the resource stream associated with this cache. */
  ResourceStream& getResourceStream() 
  {
    return m_Resources;
  }

protected:
  virtual Interface* loadNew(String& Name) = 0;
  ResourceStream& m_Resources;
  Hashtable       m_Interfaces;
};




#endif // H_WO_CACHE