/** \file DS_String.h
    Character string class

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.
Please contact the author concerning commercial usage.
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_DS_STRING
#define H_DS_STRING

#include <DS_SysDep.h>
#include <string.h>
#include <ctype.h>
#include <iostream.h>

/** A String (an array of char) with dynamic allocation.
    Provides plenty of utility methods, and operators.
    Note: Caution must be taken when constructing a string with integral
          parameters, since compilers tend to convert them to int.
          example:  String S('A');  will lead to S=="65";  ('A' ascii value) */
class String
{
public:
  /** Construct an empty string. */
  String(void);

  /** Copy constructor. */
  String(const String& S);

  /** Construct a string from a characters buffer. */
  String(const char* pBuf);

  /** Construct a string with a number.  Example:  String(123) == "123" */
  String(const int Val);

  /** Construct a string filled with the same character.  Example:  String('a',5) == "aaaaa" */
  String(const char Val, const int len);
  virtual ~String();

  /** Return a character in the string.  Characters must not be changed
      to or from a null character. */
  char& operator[](const int index);
  const char& operator[](const int index) const;

  /** Return the internal char* representation.  This is only for reading,
      to maintain the String's internal data structures. */
  const char* getString() const;

  /** Automatic conversion to c style constant string */
  operator const char*() const { return p; }

  /** Copy a string. */
  String& operator=(const String& S);

  /** Concatenate with the string and return the new created string. */
  String  operator+(const String& aString) const;

  /** Append the string with another. */
  String& operator+=(const String& S);

  friend inline DS_BOOL operator==(const String& string1, const String& string2);
  friend inline DS_BOOL operator==(const String& string1, const char*   string2);
  friend inline DS_BOOL operator==(const char*   string1, const String& string2);

  friend DS_BOOL operator<(const String& string1, const String& string2);
  friend DS_BOOL operator<(const String& string1, const char*   string2);
  friend DS_BOOL operator<(const char*   string1, const String& string2);

  friend ostream& operator<<(ostream& os, const String& s);

  friend String operator+ (const String& s, const char* c);
  friend String operator+ (const char* c, const String& s);

  /** Reads a line of input up to \n or 1024 chars. */
  static String readLine(istream& is);

  /** Returns the number of characters in the string. */
  unsigned size() const    { return Size; }

  /** Returns the number of characters in the string. */
  unsigned length() const  { return size(); }

  /** Returns the float value represented by the string. */
  float          asFloat() const;

  /** Returns the double value represented by the string. */
  double         asDouble() const;

  /** Returns the int value represented by the string. */
  long           asInt() const;

  /** Returns the int value represented by the string, assuming the string is in base 16. */
  long           asIntHex() const;

  /** Returns the unsigned int value represented by the string. */
  unsigned long  asUnsigned() const;

  /** Converts tabs to spaces.   'TabSize' spaces per tab */
  String& untabify(const int TabSize);
  /** Returns true is string consists only of digits */
  int isDigits() const;
  /** Finds the first appearance of a character.  returns -1 if not found. */
  int indexOf(const char c) const;
  /** Finds the last appearance of a character.  returns -1 if not found. */
  int lastIndexOf(const char c) const;
  /** Finds a Sub String, returns -1 if not found. */
  int indexOf(const String& S, const int CaseSense=1) const;
  /** Counts number of occurrences of a character */
  int occurrencesOf(const char c) const;
  /** Cuts the string into a substring (in place) from start (inclusive) to end (non-inclusive) */
  String& substring(const int start, const int end);
  /** Cuts the string into a substring (in place) from start (inclusive) to the end of the string */
  String& substring(const int start);
  /** Replace all occurrences of c1 by c2 */
  String& replace(const char c1, const char c2);
  /** Removes all leading and trailing spaces and tabs */
  String& trim();
  /** Converts all characters to upper case */
  String& toUpperCase();
  /** Converts all characters to lower case */
  String& toLowerCase();
  /** Removes all spaces and tabs from the string */
  String& packSpaces();
  /** Returns non zero if this string begins with the parameter string (Case sensitive) */
  int beginsWith(const String& s) const;
  /** Returns non zero if this string ends with the parameter string (Case sensitive) */
  int endsWith(const String& s) const;
  /** Returns non zero if this string is exactly the same as the parameter string (Case sensitive) */
  int equals(const String& s) const;
  /** Returns the string's crc value (used for hash values) */
  unsigned crcValue() const;

private:
  void reAlloc(const int newSize);
  void setSize(const int newSize);

  static String Hex;
  static unsigned ctab[256];

  char* p;
  int   Alloc, Size;
};


inline istream& operator>>(istream& is, String& s)
{
  s="";
  is.eatwhite();
  char buf[65];
  char c;
  do
  {
    int Cur=0;
    buf[0]='\0';
    while (!isspace(c=is.peek()))
    {
      is.get(c);
      buf[Cur++]=c;
      buf[Cur]='\0';
      if (Cur==64) break;
    }
    s+=buf;
  } while (!isspace(c));
  return is;
}

inline ostream& operator<<(ostream& os, const String& s)
{
  if (!s.p) return os;
  return os << s.p;
}

inline DS_BOOL operator==(const String& string1, const String& string2)
{ 
  if (string1.crcValue() != string2.crcValue()) return 0; // Optimization!
  return (strcmp(string1.p, string2.p) == 0); 
}

inline DS_BOOL operator==(const String& string1, const char* string2)
{
  return (strcmp(string1.p, string2) == 0); 
}

inline DS_BOOL operator==(const char* string1, const String& string2)
{ 
  return (strcmp(string1, string2.p) == 0); 
}

inline DS_BOOL operator<(const String& string1, const String& string2)
{ 
  return (strcmp(string1.p, string2.p) < 0); 
}

inline DS_BOOL operator<(const String& string1, const char* string2)
{ 
  return (strcmp(string1.p, string2) < 0); 
}

inline DS_BOOL operator<(const char* string1, const String& string2)
{ 
  return (strcmp(string1, string2.p) < 0); 
}

inline String operator+ (const String& s, const char* c)
{
  String res=s;
  return res+=c;
}

inline String operator+ (const char* c, const String& s)
{
  String res=c;
  return res+=s;
}

#endif // H_DS_STRING
