/** \file DS_SpecContainable.h
    Specialized containable classes

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.
Please contact the author concerning commercial usage.
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_DS_SPEC_CONTAINABLE
#define H_DS_SPEC_CONTAINABLE

#include <DS_Containable.h>
#include <DS_String.h>


/** A generic (still abstract) containable that represents any kind of 
    number, by using a template
    Implementation are given for 'int' and 'float' below */
template<class Number>
class Number_Containable : public Containable
{
public:
  DS_BOOL lessThan(const Containable *c) const 
  {
    return (getValue()<((Number_Containable<Number>*)c)->getValue());
  }

  DS_BOOL equals(const Containable *c) const 
  {
    return (getValue()==((Number_Containable<Number>*)c)->getValue());
  }

  size_t hashValue() const { return (size_t)getValue(); }

  virtual Number getValue() const = 0;
};

/** Abstract Implementation for 'int' subclass must provide:  int getValue() const */
typedef Number_Containable<int> Integral_Containable;

/** Non-Abstract implementation of 'int' */
class Integer_Containable : public Integral_Containable
{
public:
  Integer_Containable(const int val) : m_val(val) {}
  int getValue() const { return m_val; }
  int m_val;
};

/** Non-Abstract implementation for float */
class Float_Containable : public Number_Containable<float>
{
public:
  Float_Containable(const float val) : m_val(val) {}
  float getValue() const { return m_val; }
  float m_val;
};

/** This is a very useful subclass.  It is used for any data item
    that has no specific place in the container.
    It can even be used in hashtables, but this wouldn't be useful.
    Comparisons are done by using the object's 'this' pointer. */
class Serial_Containable : public Containable
{
public:
  DS_BOOL lessThan(const Containable *c) const { return (c!=(Containable*)this); }
  DS_BOOL equals(const Containable *c) const { return (c==(Containable*)this); }
  size_t hashValue() const { return ((size_t)this); }
};

/** An abstract string implementation, comparing lexicographically.
    Hash value is the string's crc value. */
class String_Containable : public Containable
{
public:
  DS_BOOL lessThan(const Containable *c) const { return (getString() < ((String_Containable*)c)->getString()); }
  DS_BOOL equals(const Containable *c) const { return (getString() == ((String_Containable*)c)->getString()); }
  size_t hashValue() const { return getString().crcValue(); }
  virtual String& getString() const = 0;
};

/** Non-Abstract implementation, containing an actual string */
class SimpleString_Containable : public String_Containable
{
public:
  SimpleString_Containable(const String& s) : Str(s) {}
  SimpleString_Containable(const char* c) : Str(c) {}
  String& getString() const { return (String&)Str; }
  String Str;
};

#endif // H_DS_SPEC_CONTAINABLE