/** \file 3D_Object3D.h
    3D World Object.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_3D_OBJECT3D
#define H_3D_OBJECT3D

#include <3D_World.h>
#include <CON_Sound.h>

/** 3D World Object.  Connects the concepts of a 3D World and the Mesh.
    Provides transformation, movement and collision detection for 3D objects. */
class Object3D : public Collidable_Object, public Transformable
{
public:
  /** Construct a world object.  Requires the world the object is in,
      the mesh that is associated with the object and the object's
      parent in the objects hierarchy.  The parent defaults to the 
      world in case the parameter is NULL.
  */
  Object3D(Mesh* m=NULL);
  ~Object3D();

  /** this checks to make sure that any Object added to a Object3D
      is derived from Object3D.  Returns -1 if dynamic_cast fails. */
  virtual long add(Object* O);

  /** Retrieve the mesh to be rendered for this object. */
  Mesh* getMesh();

  /** Set the mesh to be rendered for this object.  This can be NULL
      to render nothing for this object.  If not NULL, object radius is set to
      the mesh's radius. */
  void  setMesh(Mesh* m);

  /** Main rendering method.  Renders the object to the current active camera. 
      The parameter is specified as a View so that this render function will
      override the version in the base class World_Object, but the parameter
      must actually be of the View3D type. */
  virtual long render(View& view);

  /** called to determine whether to render this.  Default implementation simply
      returns !visible().  */
  virtual long preRender(View& view);

  /** Main progress method.  The Fraction indicates how much progress should be done.
      For example: If this objects advances at speed 5 units per second,
                   and Fraction=0.2   then the object should advance 5*0.2=1 unit.
      This allows keeping a synchronized behaviour, regardless to frame rate. */
  virtual long advance(float Fraction);

  /** Positions the 3D sounds the object has.  (Called by advance automatically) */
  long orientSounds();

  /** Attaches a soundclip to the object.  This must be dynamically allocated, 
      and the object will delete it automatically. */
  long startSound(SoundClip* SC, int Loop=1);

  /** Stops a soundclip that is currently playing.
      Note: Stopping the soundclip, causes it to be deleted. */
  long stopSound(SoundClip* SC);

  /** Get the object's rough radius, (non-optimized bounding sphere) */
  float getRadius() const;

  /** Set the object's radius, other than its actual one. */
  long  setRadius(const float Radius);


  // The following 3 methods determine the behaviour of the object.
  // Override them to supply specific behaviour.

  /** Returns non-zero if this object should be rendered.
      Defaults to Yes */
  virtual long visible();

  /** Returns non-zero if it should be added to the radar object, if any.
      Defaults to Yes */
  virtual long radarDetectable();

/** implement the abstract Collidable_Object functions */

  /* this tests two Object3Ds for a collision with each other */
  virtual int  isCollision(Collidable_Object* O);

  /** Provides behaviour in case of a collision with another object.
      Defaults to nothing. */
  virtual long collision(Collidable_Object* O);

  /** Returns non-zero if you want this object to be checked for collisions
      Defaults to Yes */
  virtual int  isCollidable();

protected:
  Mesh*          m_Mesh;
  float          m_Radius;
  Vector         m_Sounds;
};


class PhysicsObject3D : public Object3D
{
public:
   PhysicsObject3D(Mesh* m=NULL);

   Quat     m_rotAccel;    // rotSpeed = rotAccel * rotSpeed
   Quat     m_rotSpeed;    // orientation = rotSpeed * orientation

   float    m_mass;

   Vector3D m_thrust;      // accel = thrust / mass
   Vector3D m_speed;       // speed += accel
                           // pos += speed * orientation

  /** Applies accel vars to speed vars, then applies speed to pos/orientation.
      Final step is Object3D::advance(). */
  long advance(float Fraction);
};


/** An object which moves the camera according to its movements */
class Viewer : public PhysicsObject3D 
{
public:
  /** Construct a 3D Object and associate a camera to it. */
  Viewer(Camera *Cam, Mesh* O=NULL);

  /** Overrides the Object3D default behaviour, and renders nothing. */
  long render(const View& view);

  /** Moves the camera object to where this object is. */
  long advance(float Fraction);

  /** Returns NO, because we never see the viewer's mesh. */
  virtual long visible();

  /** Returns NO, because we don't detect ourselves. */
  virtual long radarDetectable();

  /** Returns NO, because default viewer doesn't collide. */
  virtual int  isCollidable();

protected:
  Camera* m_Cam;
};


#endif // H_3D_OBJECT3D