/*
  Telecurs.c - Automatically move the cursor to a dialog (teleport cursor).

  Jason Hood, 17 to 22 February, 2003.

  v1.10 - 6 to 9 March, 2003:
    use an array instead of a structure to store settings.
*/


#include <windows.h>
#include <shellapi.h>
#include "Telecurs.h"


#define KEY "Software\\Adoxa\\"PNAME
#define RUN "Software\\Microsoft\\Windows\\CurrentVersion\\Run"

#define WM_TRAYICON WM_USER

#ifndef IDC_LINK
#define IDC_LINK MAKEINTRESOURCE( 32649 )  // Why isn't this already defined?
#endif
#ifndef COLOR_HOTLIGHT
#define LINKCOLOR GetSysColor( 26 )	   // RGB( 0, 0, 255 ) for Win95?
#else
#define LINKCOLOR GetSysColor( COLOR_HOTLIGHT )
#endif


HINSTANCE g_hinst;
HWND	  g_hwnd;


enum { DISABLED, OVERRIDE, LOG, RET, HIDE, START, SETTINGS };
int old_settings[SETTINGS],
    settings[SETTINGS] =
{
  BST_UNCHECKED,	// Enabled
  BST_CHECKED,		// ScrollLock overrides
  BST_UNCHECKED,	// No log
  BST_CHECKED,		// Return to previous position
  BST_UNCHECKED,	// Tray icon
  BST_CHECKED,		// Start with Windows
};
char* setstr[START] = { "Disabled",
			"Override",
			"Log",
			"Return",
			"Hide" };
		      // Start is from the Run entry


Settings __declspec(dllimport) *setting;

BOOL installed = FALSE;

int  init_settings( void );
void store_settings( void );
void process_settings( void );
UINT RunEntry( UINT );

LRESULT CALLBACK WindowProc( HWND, UINT, WPARAM, LPARAM );

BOOL CALLBACK SettingsDlg( HWND, UINT, WPARAM, LPARAM );
BOOL CALLBACK AboutDlg( HWND, UINT, WPARAM, LPARAM );

void TrayAddIcon( HWND );
void TrayRemoveIcon( HWND );
void TrayProcess( HWND,	LPARAM );


// Create the main window and initialize the settings.
// If it's already running, bring up the settings dialog.
int WINAPI WinMain( HINSTANCE hInst, HINSTANCE hPrev,
		    LPSTR lpCmdLine, int nCmdShow )
{
  WNDCLASS wc;
  HWND hwnd;
  MSG  msg;
  HINSTANCE dll;
  HOOKPROC  hp;

  hwnd = FindWindow( PNAME, PNAME );
  if (hwnd)
  {
    PostMessage( hwnd, WM_COMMAND, IDM_SETTINGS, 0 );
    SetForegroundWindow( hwnd );
    return 0;
  }

  wc.style	   = 0;
  wc.lpfnWndProc   = WindowProc;
  wc.cbClsExtra    = 0;
  wc.cbWndExtra    = 0;
  wc.hInstance	   = hInst;
  wc.hIcon	   = LoadIcon( hInst, MAKEINTRESOURCE( IDI_ICON	) );
  wc.hCursor	   = LoadCursor( NULL, IDC_ARROW );
  wc.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
  wc.lpszMenuName  = NULL;
  wc.lpszClassName = PNAME;
  if (!RegisterClass( &wc )) return 0;

  hwnd = CreateWindow( PNAME, PNAME, 0, 0, 0, 0, 0, NULL, NULL, hInst, NULL );
  if (!hwnd) return 0;

  g_hinst = hInst;
  g_hwnd  = hwnd;

  if (!init_settings()) return 0;

  installed = TRUE;

  dll = LoadLibrary( "Telecurs.dll" );
  hp  = GetProcAddress( dll, "_MsgHook@12" );
  setting->hook = SetWindowsHookEx( WH_CALLWNDPROC, hp, dll, 0 );

  while (GetMessage( &msg, NULL, 0, 0 ))
  {
    TranslateMessage( &msg );
    DispatchMessage( &msg );
  }

  UnhookWindowsHookEx( setting->hook );
  FreeLibrary( dll );

  return msg.wParam;
}


// Read the settings from the registry, or bring up the dialog if this is
// the first time it's been run.
// Returns TRUE if OK was selected, FALSE if Exit or Cancel.
int init_settings( void )
{
  HKEY	key;
  DWORD exist;
  BOOL	rc = TRUE;
  int	set;

  RegCreateKeyEx( HKEY_CURRENT_USER, KEY, 0, "", REG_OPTION_NON_VOLATILE,
		  KEY_ALL_ACCESS, NULL, &key, &exist );

  if (exist == REG_CREATED_NEW_KEY)
  {
    if (DialogBox( g_hinst, MAKEINTRESOURCE( DLG_SETTINGS ),
		   g_hwnd, SettingsDlg ) != 1)
      rc = FALSE;
  }
  else
  {
    for (set = 0; set < START; ++set)
    {
      exist = sizeof(UINT);
      RegQueryValueEx( key, setstr[set], NULL, NULL, &settings[set], &exist );
    }
    settings[START] = RunEntry( -1 );
    process_settings();
  }

  RegCloseKey( key );

  exist = GetModuleFileName( NULL, setting->logfile, MAX_PATH );
  strcpy( setting->logfile + exist - 3, "LOG" );

  return rc;
}


// Write the settings into the registry.
void store_settings( void )
{
  HKEY	key;
  DWORD exist;
  int	set;

  RegCreateKeyEx( HKEY_CURRENT_USER, KEY, 0, "", REG_OPTION_NON_VOLATILE,
		  KEY_ALL_ACCESS, NULL, &key, &exist );

  for (set = 0; set < START; ++set)
    RegSetValueEx( key,setstr[set], 0,REG_DWORD, &settings[set], sizeof(UINT) );

  RunEntry( settings[START] );

  RegCloseKey( key );
}


// Pass the settings along to the hook.
void process_settings( void )
{
  if (settings[HIDE] != old_settings[HIDE] || !installed)
  {
    if (settings[HIDE] == BST_UNCHECKED) TrayAddIcon( g_hwnd );
    else if (installed) 		 TrayRemoveIcon( g_hwnd );
  }
  if (settings[RET] && !old_settings[RET])
    setting->count = 0;

  setting->disabled = settings[DISABLED];
  setting->override = settings[OVERRIDE];
  setting->log	    = settings[LOG];
  setting->ret	    = settings[RET];
}


// Retrieve, set or remove the User Run entry.
UINT RunEntry( UINT type )
{
  HKEY	run;
  DWORD len;
  UINT	rc = 0;
  char	path[MAX_PATH+2];

  RegCreateKeyEx( HKEY_CURRENT_USER, RUN, 0, "", REG_OPTION_NON_VOLATILE,
		  KEY_ALL_ACCESS, NULL, &run, &len );

  switch (type)
  {
    case BST_CHECKED:
      len = GetModuleFileName( NULL, path+1, sizeof(path)-2 );
      path[0] = path[len+1] = '\"';
      path[len+2] = 0;
      RegSetValueEx( run, PNAME, 0, REG_SZ, path, len+2 );
    break;

    case BST_UNCHECKED:
      RegDeleteValue( run, PNAME );
    break;

    default:
      rc = (RegQueryValueEx( run, PNAME, NULL, NULL, NULL, NULL )
	      == ERROR_SUCCESS) ? BST_CHECKED : BST_UNCHECKED;
    break;
  }

  RegCloseKey( run );
  return rc;
}


// Process the tray messages. The tray settings will not become the default
// settings unless the Settings dialog is OK'ed.
LRESULT CALLBACK WindowProc( HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
  switch (msg)
  {
    case WM_TRAYICON:
      TrayProcess( hwnd, lParam );
    break;

    case WM_COMMAND:
      switch (LOWORD( wParam ))
      {
	case IDM_DISABLE:
	  setting->disabled = settings[DISABLED] = !settings[DISABLED];
	break;

	case IDM_LOG:
	  setting->log = settings[LOG] = !settings[LOG];
	break;

	case IDM_ABOUT:
	  DialogBox( g_hinst, MAKEINTRESOURCE( DLG_ABOUT ), hwnd, AboutDlg );
	break;

	case IDM_SETTINGS:
	  if (DialogBox( g_hinst, MAKEINTRESOURCE( DLG_SETTINGS ), hwnd,
			 SettingsDlg ) != -1)
	    break;
	  // else fall through

	case IDM_EXIT:
	  PostMessage( hwnd, WM_DESTROY, 0, 0 );
	break;

	default: return DefWindowProc( hwnd, msg, wParam, lParam );
      }
    break;

    case WM_DESTROY:
      if (settings[HIDE] == BST_UNCHECKED) TrayRemoveIcon( hwnd );
      PostQuitMessage( 0 );
    break;

    default: return DefWindowProc( hwnd, msg, wParam, lParam );
  }
  return TRUE;
}


// Display and update the settings.
BOOL CALLBACK SettingsDlg( HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
  HICON icon;
  int	set;

  switch (msg)
  {
    case WM_INITDIALOG:
      icon = LoadIcon( g_hinst, MAKEINTRESOURCE( IDI_ICON ) );
      SetClassLong( hwnd, GCL_HICON, (LONG)icon );

      for (set = 0; set < SETTINGS; ++set)
	CheckDlgButton( hwnd, set + IDC_DISABLED, settings[set] );
    break;

    case WM_COMMAND:
      switch (LOWORD( wParam ))
      {
	case IDOK:
	  memcpy( old_settings, settings, sizeof(settings) );
	  for (set = 0; set < SETTINGS; ++set)
	    settings[set] = IsDlgButtonChecked( hwnd, set + IDC_DISABLED );
	  store_settings();
	  process_settings();
	  EndDialog( hwnd, 1 );
	break;

	case IDCANCEL:
	  EndDialog( hwnd, 0 );
	break;

	case ID_ABOUT:
	  DialogBox( g_hinst, MAKEINTRESOURCE( DLG_ABOUT ), hwnd, AboutDlg );
	break;

	case ID_EXIT:
	  // Remove the Run entry, assuming an uninstall.
	  if (IsDlgButtonChecked( hwnd, IDC_START ) == BST_UNCHECKED)
	    RunEntry( BST_UNCHECKED );
	  EndDialog( hwnd, -1 );
	break;

	default: return 0;
      }
      break;

    case WM_CLOSE:
      EndDialog( hwnd, 0 );
    break;

    default: return 0;
  }
  return 1;
}


// Display author, version and copyright info.
// It can also send me an e-mail or go to my homepage.
BOOL CALLBACK AboutDlg( HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
  HICON   icon;
  HCURSOR hand;
  HWND	  link;
  LONG	  id;

  switch (msg)
  {
    case WM_INITDIALOG:
      icon = LoadIcon( g_hinst, MAKEINTRESOURCE( IDI_ICON ) );
      SetClassLong( hwnd, GCL_HICON, (LONG)icon );

      hand = LoadCursor( NULL, IDC_LINK	);
      link = GetDlgItem( hwnd, ID_AUTHOR );
      SetClassLong( link, GCL_HCURSOR, (LONG)hand );
      //link = GetDlgItem( hwnd, ID_URL );
      //SetClassLong( link, GCL_HCURSOR, (LONG)hand );
    break;

    case WM_CTLCOLORSTATIC:
      id = GetWindowLong( (HWND)lParam, GWL_ID );
      if (id == ID_AUTHOR || id == ID_URL)
      {
	SetTextColor( (HDC)wParam, LINKCOLOR );
	SetBkMode( (HDC)wParam, TRANSPARENT );
	return (BOOL)GetSysColorBrush( COLOR_3DFACE );
      }
      return 0;
    break;

    case WM_COMMAND:
      switch (LOWORD( wParam ))
      {
	case IDOK:
	  EndDialog( hwnd, 0 );
	break;

	case ID_AUTHOR:
	case ID_URL:
	  if (HIWORD( wParam ) == STN_CLICKED)
	  {
	    ShellExecute( hwnd, NULL, (LOWORD( wParam ) == ID_AUTHOR) ?
			  "mailto:jadoxa@yahoo.com.au?Subject="PNAME
			  : PURL, NULL, ".", 0 );
	  }
	break;

	default: return 0;
      }
    break;

    case WM_CLOSE:
      EndDialog( hwnd, 0 );
    break;

    default: return 0;
  }
  return 1;
}


// Add Telecurs to the system tray.
void TrayAddIcon( HWND hwnd )
{
  NOTIFYICONDATA tray;

  tray.cbSize = sizeof(tray);
  tray.hWnd   = hwnd;
  tray.uID    = IDI_TRAY;
  tray.uFlags = NIF_ICON | NIF_MESSAGE | NIF_TIP;
  tray.uCallbackMessage = WM_TRAYICON;
  tray.hIcon  = LoadIcon( g_hinst, MAKEINTRESOURCE( IDI_ICON ) );
  strcpy( tray.szTip, PNAME );

  Shell_NotifyIcon( NIM_ADD, &tray );
}


// Remove Telecurs from the system tray.
void TrayRemoveIcon( HWND hwnd )
{
  NOTIFYICONDATA tray;

  tray.cbSize = sizeof(tray);
  tray.hWnd   = hwnd;
  tray.uID    = IDI_TRAY;
  Shell_NotifyIcon( NIM_DELETE, &tray );
}


// Process Telecurs's system tray icon.
void TrayProcess( HWND hwnd, LPARAM lParam )
{
  UINT	msg = (UINT)lParam;
  HMENU menu, popup;
  POINT pos;

  switch (msg)
  {
    case WM_LBUTTONUP:
      PostMessage( hwnd, WM_COMMAND, IDM_SETTINGS, 0 );
      SetForegroundWindow( hwnd );
    break;

    case WM_RBUTTONUP:
      menu  = LoadMenu( g_hinst, MAKEINTRESOURCE( MNU_TRAY ) );
      popup = GetSubMenu( menu, 0 );
      if (settings[DISABLED])
      {
	MENUITEMINFO mii;
	memset( &mii, 0, sizeof(mii) );
	mii.cbSize = sizeof(mii);
	mii.fMask  = MIIM_TYPE;
	mii.fType  = MFT_STRING;
	mii.wID    = IDM_DISABLE;
	mii.dwTypeData = "Enable";
	SetMenuItemInfo( menu, IDM_DISABLE, FALSE, &mii );
      }
      if (settings[LOG]) CheckMenuItem( menu, IDM_LOG, MF_CHECKED );
      GetCursorPos( &pos );
      TrackPopupMenuEx( popup, TPM_CENTERALIGN | TPM_LEFTBUTTON,
			pos.x, pos.y, hwnd, NULL );
      DestroyMenu( menu );
    break;
  }
}
